<?php

namespace Drupal\eb_aggrid\Form;

use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Access\CsrfRequestHeaderAccessCheck;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\eb\Entity\EbDefinition;
use Drupal\eb\Service\DiscoveryServiceInterface;
use Drupal\eb_aggrid\Service\FieldTypeFormSchemaBuilderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\Yaml\Yaml;

/**
 * Provides a spreadsheet-like UI form for Entity Builder definitions.
 */
class EbAggridForm extends FormBase {

  /**
   * Constructor.
   *
   * @param \Drupal\eb\Service\DiscoveryServiceInterface $discoveryService
   *   The discovery service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $tempStoreFactory
   *   The private temp store factory.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrfTokenGenerator
   *   The CSRF token generator.
   * @param \Drupal\eb_aggrid\Service\FieldTypeFormSchemaBuilderInterface $formSchemaBuilder
   *   The field type form schema builder.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    protected DiscoveryServiceInterface $discoveryService,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected PrivateTempStoreFactory $tempStoreFactory,
    protected CsrfTokenGenerator $csrfTokenGenerator,
    protected FieldTypeFormSchemaBuilderInterface $formSchemaBuilder,
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('eb.discovery_service'),
      $container->get('entity_type.manager'),
      $container->get('tempstore.private'),
      $container->get('csrf_token'),
      $container->get('eb_aggrid.field_type_form_schema_builder'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'eb_aggrid_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string|null $definition_id
   *   The definition ID if editing.
   *
   * @return array<string, mixed>
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $definition_id = NULL): array {
    // Load existing definition if editing.
    $definition = NULL;
    if ($definition_id) {
      /** @var \Drupal\eb\Entity\EbDefinition|null $definition */
      $definition = $this->entityTypeManager
        ->getStorage('eb_definition')
        ->load($definition_id);
    }

    // Defense-in-depth: verify access programmatically.
    // Routes should handle access, but this ensures form-level security.
    $this->verifyAccess($definition);

    // Add wrapper class for custom styling.
    $form['#attributes']['class'][] = 'eb-aggrid-form';

    // Attach the library and discovery data.
    $form['#attached']['library'][] = 'eb_aggrid/grid';
    // Attach field type icon libraries from core modules (only if installed).
    $icon_libraries = [
      'field_ui' => 'field_ui/drupal.field_ui.manage_fields',
      'text' => 'text/drupal.text-icon',
      'options' => 'options/drupal.options-icon',
      'file' => 'file/drupal.file-icon',
      'link' => 'link/drupal.link-icon',
      'telephone' => 'telephone/drupal.telephone-icon',
      'media' => 'media/drupal.media-icon',
      'comment' => 'comment/drupal.comment-icon',
    ];
    foreach ($icon_libraries as $module => $library) {
      if ($this->moduleHandler->moduleExists($module)) {
        $form['#attached']['library'][] = $library;
      }
    }
    $form['#attached']['drupalSettings']['ebAggrid'] = $this->getDiscoveryData($definition);

    // Conditionally attach debug library only when debug mode is active.
    if ($this->isDebugModeActive()) {
      $form['#attached']['library'][] = 'eb_aggrid/debug';
    }

    // Definition metadata.
    $form['metadata'] = [
      '#type' => 'details',
      '#title' => $this->t('Definition Metadata'),
      '#open' => TRUE,
    ];

    $form['metadata']['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#default_value' => $definition?->label() ?? '',
      '#required' => TRUE,
    ];

    $form['metadata']['definition_id'] = [
      '#type' => 'machine_name',
      '#title' => $this->t('Definition ID'),
      '#default_value' => $definition?->id() ?? '',
      '#machine_name' => [
        'exists' => [$this, 'definitionExists'],
        'source' => ['metadata', 'label'],
      ],
      '#required' => TRUE,
      '#disabled' => $definition !== NULL,
    ];

    $form['metadata']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $definition?->getDescription() ?? '',
      '#rows' => 2,
    ];

    // Toolbar with undo/redo, validation, preview, and filter (Tasks 2-4, 9).
    $form['toolbar'] = [
      '#type' => 'inline_template',
      '#template' => '
        <div class="eb-toolbar">
          <div class="eb-toolbar__group">
            <button type="button" class="eb-toolbar__btn eb-toolbar__btn--icon eb-undo-btn" title="{{ undo_title }}">
              ↶<span class="eb-toolbar__hint">Ctrl+Z</span>
            </button>
            <button type="button" class="eb-toolbar__btn eb-toolbar__btn--icon eb-redo-btn" title="{{ redo_title }}">
              ↷<span class="eb-toolbar__hint">Ctrl+Y</span>
            </button>
            <button type="button" class="eb-toolbar__btn eb-revert-btn">{{ revert_label }}</button>
          </div>
          <div class="eb-toolbar__separator"></div>
          <div class="eb-toolbar__group eb-filter-container">
            <input type="text" class="eb-quick-filter" placeholder="{{ filter_placeholder }}">
            <button type="button" class="eb-filter-clear">×</button>
            <span class="eb-filter-count"></span>
          </div>
          <div class="eb-toolbar__separator"></div>
          <div class="eb-toolbar__group">
            <div class="eb-validation-status">
              <span class="eb-validation-status__icon"></span>
              <span class="eb-validation-status__text"></span>
            </div>
          </div>
          <div class="eb-toolbar__group eb-toolbar__group--actions">
            <button type="button" class="eb-toolbar__btn eb-validate-btn">{{ validate_label }}</button>
            <button type="button" class="eb-toolbar__btn eb-preview-btn">{{ preview_label }}</button>
            {% if source_url %}<a href="{{ source_url }}" class="eb-toolbar__btn eb-source-link">{{ source_label }}</a>{% endif %}
            <button type="button" class="eb-toolbar__btn eb-toolbar__btn--icon eb-help-btn" title="{{ help_title }}">?</button>
            <button type="button" class="eb-toolbar__btn eb-toolbar__btn--icon eb-fullscreen-btn" title="{{ fullscreen_title }}">
              <span class="eb-fullscreen-btn__icon">⛶</span>
            </button>
          </div>
        </div>
      ',
      '#context' => [
        'undo_title' => $this->t('Undo (Ctrl+Z)'),
        'redo_title' => $this->t('Redo (Ctrl+Y)'),
        'revert_label' => $this->t('Revert All'),
        'filter_placeholder' => $this->t('Quick filter...'),
        'validate_label' => $this->t('Validate All'),
        'preview_label' => $this->t('Preview Changes'),
        'source_label' => $this->t('Source'),
        'source_url' => $definition ? $this->getSourceUrl($definition->id()) : NULL,
        'help_title' => $this->t('Keyboard shortcuts help'),
        'fullscreen_title' => $this->t('Toggle fullscreen (F11)'),
      ],
    ];

    // Help panel with keyboard shortcuts (Task 14.2).
    $form['help_panel'] = [
      '#type' => 'inline_template',
      '#template' => '
        <div class="eb-help-panel" style="display: none;">
          <div class="eb-help-panel__content">
            <h4>{{ shortcuts_title }}</h4>
            <div class="eb-help-panel__columns">
              <div class="eb-help-panel__column">
                <h5>{{ editing_title }}</h5>
                <dl class="eb-shortcuts">
                  <dt><kbd>Ctrl</kbd>+<kbd>Z</kbd></dt>
                  <dd>{{ undo }}</dd>
                  <dt><kbd>Ctrl</kbd>+<kbd>Y</kbd></dt>
                  <dd>{{ redo }}</dd>
                  <dt><kbd>Ctrl</kbd>+<kbd>Shift</kbd>+<kbd>N</kbd></dt>
                  <dd>{{ add_row }}</dd>
                  <dt><kbd>Delete</kbd></dt>
                  <dd>{{ delete_rows }}</dd>
                  <dt><kbd>Ctrl</kbd>+<kbd>D</kbd></dt>
                  <dd>{{ duplicate_row }}</dd>
                </dl>
              </div>
              <div class="eb-help-panel__column">
                <h5>{{ navigation_title }}</h5>
                <dl class="eb-shortcuts">
                  <dt><kbd>↑</kbd> <kbd>↓</kbd> <kbd>←</kbd> <kbd>→</kbd></dt>
                  <dd>{{ navigate }}</dd>
                  <dt><kbd>Enter</kbd></dt>
                  <dd>{{ start_edit }}</dd>
                  <dt><kbd>Escape</kbd></dt>
                  <dd>{{ cancel_edit }}</dd>
                  <dt><kbd>Tab</kbd></dt>
                  <dd>{{ next_cell }}</dd>
                  <dt><kbd>Shift</kbd>+<kbd>Tab</kbd></dt>
                  <dd>{{ prev_cell }}</dd>
                </dl>
              </div>
              <div class="eb-help-panel__column">
                <h5>{{ selection_title }}</h5>
                <dl class="eb-shortcuts">
                  <dt><kbd>Ctrl</kbd>+<kbd>A</kbd></dt>
                  <dd>{{ select_all }}</dd>
                  <dt><kbd>Shift</kbd>+<kbd>↑</kbd>/<kbd>↓</kbd></dt>
                  <dd>{{ extend_selection }}</dd>
                  <dt><kbd>Ctrl</kbd>+<kbd>Click</kbd></dt>
                  <dd>{{ toggle_row }}</dd>
                </dl>
              </div>
            </div>
            <p class="eb-help-panel__note">{{ mac_note }}</p>
          </div>
        </div>
      ',
      '#context' => [
        'shortcuts_title' => $this->t('Keyboard Shortcuts'),
        'editing_title' => $this->t('Editing'),
        'navigation_title' => $this->t('Navigation'),
        'selection_title' => $this->t('Selection'),
        'undo' => $this->t('Undo last change'),
        'redo' => $this->t('Redo last undone change'),
        'add_row' => $this->t('Add new row'),
        'delete_rows' => $this->t('Delete selected rows'),
        'duplicate_row' => $this->t('Duplicate selected row'),
        'navigate' => $this->t('Navigate between cells'),
        'start_edit' => $this->t('Start editing cell'),
        'cancel_edit' => $this->t('Cancel editing'),
        'next_cell' => $this->t('Move to next cell'),
        'prev_cell' => $this->t('Move to previous cell'),
        'select_all' => $this->t('Select all rows'),
        'extend_selection' => $this->t('Extend selection'),
        'toggle_row' => $this->t('Toggle row selection'),
        'mac_note' => $this->t('On Mac, use Cmd instead of Ctrl'),
      ],
    ];

    // Spreadsheet container with Excel-like bottom tabs.
    $form['spreadsheet_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['eb-spreadsheet-container'],
      ],
    ];

    // Tab content panels - all rendered but only one visible at a time.
    $form['spreadsheet_container']['tab_panels'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['eb-tab-panels'],
      ],
    ];

    // Bundle definitions panel.
    $form['spreadsheet_container']['tab_panels']['bundles'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['eb-tab-panel', 'eb-tab-panel--active'],
        'data-tab' => 'bundle_definitions',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['bundles']['bundle_grid_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'eb-bundle-grid',
        'class' => ['ag-theme-material', 'eb-grid'],
        'style' => 'height: 400px; width: 100%;',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['bundles']['bundle_definitions_data'] = [
      '#type' => 'hidden',
      '#default_value' => $definition ? json_encode($definition->getBundleDefinitions()) : '[]',
      '#attributes' => ['id' => 'bundle-definitions-data'],
    ];

    $form['spreadsheet_container']['tab_panels']['bundles']['bundle_actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['grid-actions']],
    ];

    $form['spreadsheet_container']['tab_panels']['bundles']['bundle_actions']['add_bundle'] = [
      '#type' => 'button',
      '#value' => $this->t('Add Bundle'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-grid-action' => 'add-row',
        'data-grid-target' => 'bundle',
      ],
      '#limit_validation_errors' => [],
    ];

    // Field definitions panel.
    $form['spreadsheet_container']['tab_panels']['fields'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['eb-tab-panel'],
        'data-tab' => 'field_definitions',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['fields']['field_grid_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'eb-field-grid',
        'class' => ['ag-theme-material', 'eb-grid'],
        'style' => 'height: 400px; width: 100%;',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['fields']['field_definitions_data'] = [
      '#type' => 'hidden',
      '#default_value' => $definition ? json_encode($definition->getFieldDefinitions()) : '[]',
      '#attributes' => ['id' => 'field-definitions-data'],
    ];

    $form['spreadsheet_container']['tab_panels']['fields']['field_actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['grid-actions']],
    ];

    $form['spreadsheet_container']['tab_panels']['fields']['field_actions']['add_field'] = [
      '#type' => 'button',
      '#value' => $this->t('Add Field'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-grid-action' => 'add-row',
        'data-grid-target' => 'field',
      ],
      '#limit_validation_errors' => [],
    ];

    // Field group definitions panel.
    $form['spreadsheet_container']['tab_panels']['field_groups'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['eb-tab-panel'],
        'data-tab' => 'field_group_definitions',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['field_groups']['field_group_grid_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'eb-field-group-grid',
        'class' => ['ag-theme-material', 'eb-grid'],
        'style' => 'height: 400px; width: 100%;',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['field_groups']['field_group_definitions_data'] = [
      '#type' => 'hidden',
      '#default_value' => $definition ? json_encode($definition->getFieldGroupDefinitions()) : '[]',
      '#attributes' => ['id' => 'field-group-definitions-data'],
    ];

    $form['spreadsheet_container']['tab_panels']['field_groups']['field_group_actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['grid-actions']],
    ];

    $form['spreadsheet_container']['tab_panels']['field_groups']['field_group_actions']['add_field_group'] = [
      '#type' => 'button',
      '#value' => $this->t('Add Field Group'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-grid-action' => 'add-row',
        'data-grid-target' => 'field_group',
      ],
      '#limit_validation_errors' => [],
    ];

    // Display field definitions panel.
    $form['spreadsheet_container']['tab_panels']['displays'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['eb-tab-panel'],
        'data-tab' => 'display_field_definitions',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['displays']['display_grid_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'eb-display-grid',
        'class' => ['ag-theme-material', 'eb-grid'],
        'style' => 'height: 400px; width: 100%;',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['displays']['display_field_definitions_data'] = [
      '#type' => 'hidden',
      '#default_value' => $definition ? json_encode($definition->getDisplayFieldDefinitions()) : '[]',
      '#attributes' => ['id' => 'display-field-definitions-data'],
    ];

    $form['spreadsheet_container']['tab_panels']['displays']['display_actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['grid-actions']],
    ];

    $form['spreadsheet_container']['tab_panels']['displays']['display_actions']['add_display'] = [
      '#type' => 'button',
      '#value' => $this->t('Add Display Field'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-grid-action' => 'add-row',
        'data-grid-target' => 'display',
      ],
      '#limit_validation_errors' => [],
    ];

    // Menu definitions panel.
    $form['spreadsheet_container']['tab_panels']['menus'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['eb-tab-panel'],
        'data-tab' => 'menu_definitions',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['menus']['menu_grid_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'eb-menu-grid',
        'class' => ['ag-theme-material', 'eb-grid'],
        'style' => 'height: 400px; width: 100%;',
      ],
    ];

    $form['spreadsheet_container']['tab_panels']['menus']['menu_definitions_data'] = [
      '#type' => 'hidden',
      '#default_value' => $definition ? json_encode($definition->getMenuDefinitions()) : '[]',
      '#attributes' => ['id' => 'menu-definitions-data'],
    ];

    $form['spreadsheet_container']['tab_panels']['menus']['menu_actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['grid-actions']],
    ];

    $form['spreadsheet_container']['tab_panels']['menus']['menu_actions']['add_menu'] = [
      '#type' => 'button',
      '#value' => $this->t('Add Menu'),
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-grid-action' => 'add-row',
        'data-grid-target' => 'menu',
      ],
      '#limit_validation_errors' => [],
    ];

    // Resize handle for adjustable grid height.
    $form['spreadsheet_container']['resize_handle'] = [
      '#type' => 'inline_template',
      '#template' => '<div class="eb-resize-handle" title="{{ title }}" role="slider" aria-label="{{ aria_label }}" aria-valuemin="300" aria-valuemax="1500"></div>',
      '#context' => [
        'title' => $this->t('Drag to resize'),
        'aria_label' => $this->t('Resize grid height'),
      ],
    ];

    // Bottom tab bar (Excel-like sheet tabs).
    $form['spreadsheet_container']['tab_bar'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['eb-tab-bar'],
      ],
    ];

    $form['spreadsheet_container']['tab_bar']['tabs'] = [
      '#type' => 'inline_template',
      '#template' => '
        <div class="eb-sheet-tabs">
          <button type="button" class="eb-sheet-tab eb-sheet-tab--active" data-tab="bundle_definitions">
            <span class="eb-sheet-tab__icon">📦</span>
            <span class="eb-sheet-tab__label">Bundles</span>
          </button>
          <button type="button" class="eb-sheet-tab" data-tab="field_definitions">
            <span class="eb-sheet-tab__icon">📝</span>
            <span class="eb-sheet-tab__label">Fields</span>
          </button>
          <button type="button" class="eb-sheet-tab" data-tab="field_group_definitions">
            <span class="eb-sheet-tab__icon">📁</span>
            <span class="eb-sheet-tab__label">Field Groups</span>
          </button>
          <button type="button" class="eb-sheet-tab" data-tab="display_field_definitions">
            <span class="eb-sheet-tab__icon">🖼️</span>
            <span class="eb-sheet-tab__label">Displays</span>
          </button>
          <button type="button" class="eb-sheet-tab" data-tab="menu_definitions">
            <span class="eb-sheet-tab__icon">☰</span>
            <span class="eb-sheet-tab__label">Menus</span>
          </button>
        </div>
      ',
    ];

    // Form actions.
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Definition'),
      '#button_type' => 'primary',
    ];

    $form['actions']['export_yaml'] = [
      '#type' => 'submit',
      '#value' => $this->t('Export YAML'),
      '#submit' => ['::exportYaml'],
      '#limit_validation_errors' => [],
    ];

    // Only show "Save and Apply" if user has apply access.
    if ($definition && $definition->access('apply')) {
      $form['actions']['apply'] = [
        '#type' => 'submit',
        '#value' => $this->t('Save and Apply'),
        '#submit' => ['::saveAndApply'],
      ];
    }

    return $form;
  }

  /**
   * Gets discovery data for JavaScript.
   *
   * @param \Drupal\eb\Entity\EbDefinition|null $definition
   *   The existing definition or NULL.
   *
   * @return array<string, mixed>
   *   Discovery data for drupalSettings.
   */
  protected function getDiscoveryData(?EbDefinition $definition): array {
    // Get entity types, their labels, and bundle labels.
    $entity_types = [];
    $entity_type_labels = [];
    $bundle_labels = [];
    foreach ($this->discoveryService->getSupportedEntityTypes() as $entity_type_id => $entity_type) {
      $entity_types[] = $entity_type_id;
      $entity_type_labels[$entity_type_id] = $entity_type['label'] ?? $entity_type_id;
      // Collect bundle labels for each entity type from discovery service.
      $bundle_labels[$entity_type_id] = [];
      foreach ($this->discoveryService->getBundlesForEntityType($entity_type_id) as $bundle_id => $bundle_info) {
        $bundle_labels[$entity_type_id][$bundle_id] = $bundle_info['label'] ?? $bundle_id;
      }
    }

    // Get field types with labels, descriptions, and categories.
    $all_field_types = $this->discoveryService->getAllFieldTypes();
    $field_types = array_keys($all_field_types);
    $field_type_labels = [];
    $field_type_descriptions = [];
    $field_type_categories = [];
    foreach ($all_field_types as $type_id => $type_info) {
      $field_type_labels[$type_id] = $type_info['label'] ?? $type_id;
      $field_type_descriptions[$type_id] = $type_info['description'] ?? '';
      $field_type_categories[$type_id] = $type_info['category'] ?? 'General';
    }

    // Get default settings for each field type (from form schema builder).
    $field_type_default_settings = [];
    foreach ($field_types as $field_type) {
      $field_type_default_settings[$field_type] = $this->formSchemaBuilder->getDefaultSettings($field_type);
    }

    // Get widgets with labels and default settings.
    $widgets = [];
    foreach ($this->discoveryService->getAllWidgets() as $widget_id => $widget_info) {
      $widgets[$widget_id] = [
        'label' => $widget_info['label'] ?? $widget_id,
        'default_settings' => $widget_info['default_settings'] ?? [],
      ];
    }

    // Get formatters with labels and default settings.
    $formatters = [];
    foreach ($this->discoveryService->getAllFormatters() as $formatter_id => $formatter_info) {
      $formatters[$formatter_id] = [
        'label' => $formatter_info['label'] ?? $formatter_id,
        'default_settings' => $formatter_info['default_settings'] ?? [],
      ];
    }

    // Get widgets by field type for dynamic filtering.
    $widgets_by_field_type = [];
    foreach ($field_types as $field_type) {
      $widgets_by_field_type[$field_type] = array_keys(
        $this->discoveryService->getWidgetsForFieldType($field_type)
      );
    }

    // Get formatters by field type for dynamic filtering.
    $formatters_by_field_type = [];
    foreach ($field_types as $field_type) {
      $formatters_by_field_type[$field_type] = array_keys(
        $this->discoveryService->getFormattersForFieldType($field_type)
      );
    }

    // Field group format types.
    $field_group_format_types = [
      'fieldset',
      'details',
      'tabs',
      'tab',
      'accordion',
      'accordion_item',
      'html_element',
    ];

    // Get available text formats for formatted text fields.
    $text_formats = [];
    $format_storage = $this->entityTypeManager->getStorage('filter_format');
    /** @var \Drupal\filter\FilterFormatInterface $format */
    foreach ($format_storage->loadMultiple() as $format_id => $format) {
      $text_formats[$format_id] = $format->label();
    }

    // Build extensions data for enabled extension modules.
    $extensions = [];

    // Add field_group extension data if the module is enabled.
    if ($this->moduleHandler->moduleExists('eb_field_group')) {
      $field_groups = $definition?->getFieldGroupDefinitions() ?? [];
      $form_groups = [];
      $view_groups = [];

      foreach ($field_groups as $group) {
        if (empty($group['group_name'])) {
          continue;
        }

        $display_type = $group['display_type'] ?? '';
        $mode = $group['mode'] ?? '';

        // Only include default mode groups for form_group/view_group columns.
        if ($mode !== 'default') {
          continue;
        }

        $item = [
          'name' => $group['group_name'],
          'label' => $group['label'] ?? $group['group_name'],
        ];

        if ($display_type === 'form') {
          $form_groups[] = $item;
        }
        elseif ($display_type === 'view') {
          $view_groups[] = $item;
        }
      }

      $extensions['field_group'] = [
        'enabled' => TRUE,
        'formGroups' => $form_groups,
        'viewGroups' => $view_groups,
      ];
    }
    else {
      // Provide disabled state for field_group so UI can show disabled tab.
      $extensions['field_group'] = [
        'enabled' => FALSE,
        'formGroups' => [],
        'viewGroups' => [],
      ];
    }

    // Add pathauto extension status.
    $extensions['pathauto'] = [
      'enabled' => $this->moduleHandler->moduleExists('eb_pathauto'),
    ];

    // Add auto_entitylabel extension status.
    $extensions['auto_entitylabel'] = [
      'enabled' => $this->moduleHandler->moduleExists('eb_auto_entitylabel'),
    ];

    return [
      'discovery' => [
        'entityTypes' => $entity_types,
        'entityTypeLabels' => $entity_type_labels,
        'bundleLabels' => $bundle_labels,
        'fieldTypes' => $field_types,
        'fieldTypeLabels' => $field_type_labels,
        'fieldTypeDescriptions' => $field_type_descriptions,
        'fieldTypeCategories' => $field_type_categories,
        'fieldTypeDefaultSettings' => $field_type_default_settings,
        'fieldTypeFormSchemas' => $this->formSchemaBuilder->getAllFormSchemas(),
        'widgets' => $widgets,
        'formatters' => $formatters,
        'widgetsByFieldType' => $widgets_by_field_type,
        'formattersByFieldType' => $formatters_by_field_type,
        'fieldGroupFormatTypes' => $field_group_format_types,
        'textFormats' => $text_formats,
        'displayTypes' => ['form', 'view'],
        'displayModes' => ['default'],
        'formDisplayModes' => $this->getFormDisplayModes(),
        'viewDisplayModes' => $this->getViewDisplayModes(),
      ],
      'bundleDefinitions' => $definition?->getBundleDefinitions() ?? [],
      'fieldDefinitions' => $definition?->getFieldDefinitions() ?? [],
      'fieldGroupDefinitions' => $definition?->getFieldGroupDefinitions() ?? [],
      'displayFieldDefinitions' => $definition?->getDisplayFieldDefinitions() ?? [],
      'menuDefinitions' => $definition?->getMenuDefinitions() ?? [],
      // CSRF token for AJAX requests.
      // Must use same key as CsrfRequestHeaderAccessCheck for validation.
      'csrfToken' => $this->csrfTokenGenerator->get(CsrfRequestHeaderAccessCheck::TOKEN_KEY),
      // UI timing settings from configuration.
      'settings' => [
        'validation_debounce_ms' => $this->configFactory()->get('eb_aggrid.settings')->get('validation_debounce_ms'),
        'sync_debounce_ms' => $this->configFactory()->get('eb_aggrid.settings')->get('sync_debounce_ms'),
      ],
      // Debug mode: enabled via URL param if config allows it.
      'debugMode' => $this->isDebugModeActive(),
      // Extension-specific data (e.g., field_group options).
      'extensions' => $extensions,
    ];
  }

  /**
   * Determines if debug mode should be active.
   *
   * Debug mode is active when:
   * 1. The config setting 'debug_mode_enabled' is TRUE, AND
   * 2. The URL query parameter 'eb_debug' equals '1', AND
   * 3. The current user has 'administer site configuration' permission.
   *
   * @return bool
   *   TRUE if debug mode is active, FALSE otherwise.
   */
  protected function isDebugModeActive(): bool {
    $config = $this->configFactory()->get('eb_aggrid.settings');
    $debug_enabled = (bool) $config->get('debug_mode_enabled');

    if (!$debug_enabled) {
      return FALSE;
    }

    // Security: require admin permission for debug mode.
    if (!$this->currentUser()->hasPermission('administer site configuration')) {
      return FALSE;
    }

    $request = $this->getRequest();
    return $request->query->get('eb_debug') === '1';
  }

  /**
   * Checks if a definition ID already exists.
   *
   * @param string $value
   *   The machine name value.
   *
   * @return bool
   *   TRUE if the definition exists, FALSE otherwise.
   */
  public function definitionExists(string $value): bool {
    $definition = $this->entityTypeManager
      ->getStorage('eb_definition')
      ->load($value);
    return $definition !== NULL;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Decode and validate the hidden JSON data.
    $bundle_data = json_decode($form_state->getValue('bundle_definitions_data') ?? '[]', TRUE);
    $field_data = json_decode($form_state->getValue('field_definitions_data') ?? '[]', TRUE);

    if (!is_array($bundle_data)) {
      $form_state->setErrorByName('bundle_definitions_data', $this->t('Invalid bundle definitions data.'));
    }

    if (!is_array($field_data)) {
      $form_state->setErrorByName('field_definitions_data', $this->t('Invalid field definitions data.'));
    }
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $definition_id = $form_state->getValue('definition_id');

    // Load or create the definition.
    $storage = $this->entityTypeManager->getStorage('eb_definition');
    /** @var \Drupal\eb\Entity\EbDefinition|null $definition */
    $definition = $storage->load($definition_id);

    if (!$definition) {
      /** @var \Drupal\eb\Entity\EbDefinition $definition */
      $definition = $storage->create(['id' => $definition_id]);
    }

    // Set metadata with sanitization.
    $label = $form_state->getValue('label');
    $description = $form_state->getValue('description') ?? '';
    $definition->set('label', Xss::filter($label));
    $definition->setDescription(Xss::filter($description));

    // Set definition data from grids with sanitization.
    $definition->setBundleDefinitions(
      $this->sanitizeGridData(
        json_decode($form_state->getValue('bundle_definitions_data') ?? '[]', TRUE) ?: []
      )
    );
    $definition->setFieldDefinitions(
      $this->sanitizeGridData(
        json_decode($form_state->getValue('field_definitions_data') ?? '[]', TRUE) ?: []
      )
    );
    $definition->setFieldGroupDefinitions(
      $this->sanitizeGridData(
        json_decode($form_state->getValue('field_group_definitions_data') ?? '[]', TRUE) ?: []
      )
    );
    $definition->setDisplayFieldDefinitions(
      $this->sanitizeGridData(
        json_decode($form_state->getValue('display_field_definitions_data') ?? '[]', TRUE) ?: []
      )
    );
    $definition->setMenuDefinitions(
      $this->sanitizeGridData(
        json_decode($form_state->getValue('menu_definitions_data') ?? '[]', TRUE) ?: []
      )
    );

    $definition->save();

    $this->messenger()->addStatus($this->t('Definition "@label" has been saved.', [
      '@label' => $definition->label(),
    ]));

    $form_state->setRedirect('eb_aggrid.edit', [
      'definition_id' => $definition_id,
    ]);
  }

  /**
   * Fields that are machine names and should use strict escaping.
   *
   * These fields should only contain alphanumeric characters, underscores,
   * and hyphens. Using Html::escape ensures no HTML is allowed.
   *
   * @var array<string>
   */
  protected const MACHINE_NAME_FIELDS = [
    'entity_type',
    'bundle',
    'bundle_id',
    'field_name',
    'field_type',
    'group_name',
    'format_type',
    'widget',
    'formatter',
    'menu_id',
    'mode',
    'display_type',
  ];

  /**
   * Sanitize grid data to prevent XSS attacks.
   *
   * Uses Html::escape() for machine name fields (no HTML allowed) and
   * Xss::filter() for descriptive fields (safe HTML tags allowed).
   *
   * @param array<mixed> $data
   *   The grid data to sanitize.
   * @param int $depth
   *   Current recursion depth.
   *
   * @return array<mixed>
   *   The sanitized data.
   */
  protected function sanitizeGridData(array $data, int $depth = 0): array {
    // Prevent infinite recursion.
    if ($depth > 10) {
      return [];
    }

    $sanitized = [];
    foreach ($data as $key => $value) {
      if (is_string($value)) {
        // Use strict escaping for machine name fields.
        if (in_array($key, self::MACHINE_NAME_FIELDS, TRUE)) {
          $sanitized[$key] = Html::escape($value);
        }
        else {
          // Allow safe HTML tags in descriptive fields.
          $sanitized[$key] = Xss::filter($value);
        }
      }
      elseif (is_array($value)) {
        // Recursively sanitize nested arrays.
        $sanitized[$key] = $this->sanitizeGridData($value, $depth + 1);
      }
      else {
        // Keep other types (int, bool, null) as-is.
        $sanitized[$key] = $value;
      }
    }

    return $sanitized;
  }

  /**
   * Export YAML submit handler.
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function exportYaml(array &$form, FormStateInterface $form_state): void {
    // Build YAML structure from form data.
    $yaml = [
      'version' => EbDefinition::FORMAT_VERSION,
      'mode' => 'sync',
      'scope' => 'partial',
      'id' => $form_state->getValue('definition_id'),
      'label' => $form_state->getValue('label'),
    ];

    $description = $form_state->getValue('description');
    if (!empty($description)) {
      $yaml['description'] = $description;
    }

    $bundle_definitions = json_decode($form_state->getValue('bundle_definitions_data') ?? '[]', TRUE);
    if (!empty($bundle_definitions)) {
      $yaml['bundle_definitions'] = $bundle_definitions;
    }

    $field_definitions = json_decode($form_state->getValue('field_definitions_data') ?? '[]', TRUE);
    if (!empty($field_definitions)) {
      $yaml['field_definitions'] = $field_definitions;
    }

    $field_group_definitions = json_decode($form_state->getValue('field_group_definitions_data') ?? '[]', TRUE);
    if (!empty($field_group_definitions)) {
      $yaml['field_group_definitions'] = $field_group_definitions;
    }

    $display_field_definitions = json_decode($form_state->getValue('display_field_definitions_data') ?? '[]', TRUE);
    if (!empty($display_field_definitions)) {
      $yaml['display_field_definitions'] = $display_field_definitions;
    }

    $menu_definitions = json_decode($form_state->getValue('menu_definitions_data') ?? '[]', TRUE);
    if (!empty($menu_definitions)) {
      $yaml['menu_definitions'] = $menu_definitions;
    }

    // Convert to YAML and trigger download.
    $yaml_content = Yaml::dump($yaml, 10, 2);

    // Store in session for download.
    $tempstore = $this->tempStoreFactory->get('eb_aggrid');
    $tempstore->set('yaml_export', [
      'content' => $yaml_content,
      'filename' => ($form_state->getValue('definition_id') ?: 'definition') . '.yml',
    ]);

    $this->messenger()->addStatus($this->t('YAML generated. Use the browser download feature to save it.'));
  }

  /**
   * Save and apply submit handler.
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function saveAndApply(array &$form, FormStateInterface $form_state): void {
    // First save the definition.
    $this->submitForm($form, $form_state);

    // Then redirect to apply form.
    $definition_id = $form_state->getValue('definition_id');
    $form_state->setRedirect('entity.eb_definition.apply_form', [
      'eb_definition' => $definition_id,
    ]);
  }

  /**
   * Gets available form display modes.
   *
   * @return array<string>
   *   Array of form display mode IDs.
   */
  protected function getFormDisplayModes(): array {
    $modes = ['default'];

    try {
      $storage = $this->entityTypeManager->getStorage('entity_form_mode');
      foreach ($storage->loadMultiple() as $mode) {
        $mode_id = $mode->id();
        // Mode ID format is "entity_type.mode_name", extract just the mode name.
        $parts = explode('.', $mode_id);
        $mode_name = end($parts);
        if (!in_array($mode_name, $modes, TRUE)) {
          $modes[] = $mode_name;
        }
      }
    }
    catch (\Exception $e) {
      // Fall back to default if entity type doesn't exist.
    }

    return $modes;
  }

  /**
   * Gets available view display modes.
   *
   * @return array<string>
   *   Array of view display mode IDs.
   */
  protected function getViewDisplayModes(): array {
    $modes = ['default', 'teaser', 'full'];

    try {
      $storage = $this->entityTypeManager->getStorage('entity_view_mode');
      foreach ($storage->loadMultiple() as $mode) {
        $mode_id = $mode->id();
        // Mode ID format is "entity_type.mode_name", extract just the mode name.
        $parts = explode('.', $mode_id);
        $mode_name = end($parts);
        if (!in_array($mode_name, $modes, TRUE)) {
          $modes[] = $mode_name;
        }
      }
    }
    catch (\Exception $e) {
      // Fall back to defaults if entity type doesn't exist.
    }

    return array_unique($modes);
  }

  /**
   * Gets the source URL for a definition.
   *
   * @param string $definition_id
   *   The definition ID.
   *
   * @return string
   *   The source URL.
   */
  protected function getSourceUrl(string $definition_id): string {
    return Url::fromRoute('eb_aggrid.source', [
      'definition_id' => $definition_id,
    ])->toString();
  }

  /**
   * Verify access to the form (defense-in-depth).
   *
   * Routes should handle access control, but this provides an additional
   * layer of security to ensure users have the proper permissions.
   *
   * @param \Drupal\eb\Entity\EbDefinition|null $definition
   *   The definition entity, or NULL for new definitions.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   If the user does not have access.
   */
  protected function verifyAccess(?EbDefinition $definition): void {
    $access_handler = $this->entityTypeManager
      ->getAccessControlHandler('eb_definition');

    if ($definition === NULL) {
      // Creating new definition - check create access.
      if (!$access_handler->createAccess()) {
        throw new AccessDeniedHttpException('You do not have permission to create entity definitions.');
      }
    }
    else {
      // Editing existing definition - check update access.
      if (!$definition->access('update')) {
        throw new AccessDeniedHttpException('You do not have permission to edit this entity definition.');
      }
    }
  }

}
