<?php

namespace Drupal\eb_aggrid\Form;

use Drupal\Core\Form\FormStateInterface;

/**
 * Provides grid settings integration for the Entity Builder settings form.
 *
 * This class handles the form alter to add vertical tabs and grid settings
 * to the core Entity Builder settings form.
 */
class EbAggridSettingsForm {

  /**
   * Default grid settings values.
   */
  protected const DEFAULTS = [
    'validation_debounce_ms' => 500,
    'sync_debounce_ms' => 150,
    'max_json_depth' => 10,
    'max_content_size' => 5,
    'debug_mode_enabled' => FALSE,
  ];

  /**
   * Alters the eb_settings form.
   *
   * Adds vertical tabs and grid settings to the core settings form.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function alter(array &$form, FormStateInterface $form_state): void {
    // Add vertical tabs container.
    $form['settings_tabs'] = [
      '#type' => 'vertical_tabs',
      '#default_tab' => 'edit-core-settings',
      '#weight' => -100,
    ];

    // Create Core Settings tab and move existing form elements into it.
    $form['core_settings'] = [
      '#type' => 'details',
      '#title' => t('Core Settings'),
      '#group' => 'settings_tabs',
      '#weight' => 0,
    ];

    // Move existing form groups into Core Settings tab.
    $core_groups = ['general', 'import', 'rollback', 'pagination', 'limits'];
    foreach ($core_groups as $group) {
      if (isset($form[$group])) {
        $form['core_settings'][$group] = $form[$group];
        unset($form[$group]);
      }
    }

    // Create Grid Settings tab.
    $form['grid_settings'] = [
      '#type' => 'details',
      '#title' => t('Grid Settings'),
      '#group' => 'settings_tabs',
      '#weight' => 10,
    ];

    // Load Grid config.
    $grid_config = \Drupal::config('eb_aggrid.settings');

    // Grid Performance section.
    $form['grid_settings']['grid_performance'] = [
      '#type' => 'details',
      '#title' => t('Grid Performance'),
      '#open' => TRUE,
    ];

    $form['grid_settings']['grid_performance']['validation_debounce_ms'] = [
      '#type' => 'number',
      '#title' => t('Validation debounce'),
      '#description' => t('Delay before triggering validation after user input. Lower values provide faster feedback but increase server load.'),
      '#default_value' => $grid_config->get('validation_debounce_ms') ?? self::DEFAULTS['validation_debounce_ms'],
      '#min' => 100,
      '#max' => 2000,
      '#field_suffix' => t('ms'),
    ];

    $form['grid_settings']['grid_performance']['sync_debounce_ms'] = [
      '#type' => 'number',
      '#title' => t('Sync debounce'),
      '#description' => t('Delay before triggering data synchronization after user input.'),
      '#default_value' => $grid_config->get('sync_debounce_ms') ?? self::DEFAULTS['sync_debounce_ms'],
      '#min' => 50,
      '#max' => 1000,
      '#field_suffix' => t('ms'),
    ];

    // AJAX Security Limits section.
    $form['grid_settings']['ajax_limits'] = [
      '#type' => 'details',
      '#title' => t('AJAX Security Limits'),
      '#open' => TRUE,
    ];

    $form['grid_settings']['ajax_limits']['max_json_depth'] = [
      '#type' => 'number',
      '#title' => t('Maximum JSON depth'),
      '#description' => t('Maximum nesting depth for JSON parsing in AJAX requests. Limits prevent denial of service attacks.'),
      '#default_value' => $grid_config->get('max_json_depth') ?? self::DEFAULTS['max_json_depth'],
      '#min' => 5,
      '#max' => 50,
    ];

    $form['grid_settings']['ajax_limits']['max_content_size'] = [
      '#type' => 'number',
      '#title' => t('Maximum AJAX content size'),
      '#description' => t('Maximum size of AJAX request content. Limits prevent memory exhaustion.'),
      '#default_value' => $grid_config->get('max_content_size') ?? self::DEFAULTS['max_content_size'],
      '#min' => 1,
      '#max' => 10,
      '#step' => 1,
      '#field_suffix' => t('MB'),
    ];

    // Debug Settings section.
    $form['grid_settings']['debug_settings'] = [
      '#type' => 'details',
      '#title' => t('Debug Settings'),
      '#open' => FALSE,
    ];

    $form['grid_settings']['debug_settings']['debug_mode_enabled'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable debug mode via URL parameter'),
      '#description' => t('When enabled, append <code>?eb_debug=1</code> to the Entity Builder form URL to show expandable debug panels for each grid row. Debug panels display raw field data, settings, and validation state.'),
      '#default_value' => $grid_config->get('debug_mode_enabled') ?? self::DEFAULTS['debug_mode_enabled'],
    ];

    // Add custom submit handler for grid settings.
    $form['#submit'][] = [static::class, 'submit'];
  }

  /**
   * Submit handler for grid settings.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function submit(array &$form, FormStateInterface $form_state): void {
    \Drupal::configFactory()->getEditable('eb_aggrid.settings')
      ->set('validation_debounce_ms', (int) $form_state->getValue('validation_debounce_ms'))
      ->set('sync_debounce_ms', (int) $form_state->getValue('sync_debounce_ms'))
      ->set('max_json_depth', (int) $form_state->getValue('max_json_depth'))
      ->set('max_content_size', (int) $form_state->getValue('max_content_size'))
      ->set('debug_mode_enabled', (bool) $form_state->getValue('debug_mode_enabled'))
      ->save();
  }

}
