<?php

namespace Drupal\eb_aggrid\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Builds JSON form schemas for field type settings.
 *
 * This service dynamically generates form schemas by:
 * 1. Reading default settings from Drupal's field type classes
 * 2. Filtering out internal/complex settings via a blocklist
 * 3. Applying human-friendly enhancements (labels, descriptions, options)
 * 4. Supporting contrib modules via hook_alter.
 */
class FieldTypeFormSchemaBuilder implements FieldTypeFormSchemaBuilderInterface {

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Field\FieldTypePluginManagerInterface $fieldTypeManager
   *   The field type plugin manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler for hook_alter support.
   */
  public function __construct(
    protected FieldTypePluginManagerInterface $fieldTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getFormSchema(string $fieldType): array {
    $definition = $this->fieldTypeManager->getDefinition($fieldType, FALSE);
    if ($definition === NULL) {
      return [];
    }

    $class = $definition['class'] ?? NULL;
    if ($class === NULL || !class_exists($class)) {
      return [];
    }

    $schema = [
      'storage' => [],
      'config' => [],
    ];

    // Get default values from field type class.
    $storageDefaults = method_exists($class, 'defaultStorageSettings')
      ? $class::defaultStorageSettings() : [];
    $configDefaults = method_exists($class, 'defaultFieldSettings')
      ? $class::defaultFieldSettings() : [];

    // Get enhancements for this field type.
    $enhancements = $this->getEnhancements($fieldType);

    // Build schema from storage defaults, filtering out internal settings.
    foreach ($storageDefaults as $key => $default) {
      if ($this->shouldExpose($key, $default, $fieldType, 'storage')) {
        $schema['storage'][$key] = $this->buildFieldSchema($key, $default, $fieldType, 'storage', $enhancements);
      }
    }

    // Build schema from config defaults, filtering out internal settings.
    foreach ($configDefaults as $key => $default) {
      if ($this->shouldExpose($key, $default, $fieldType, 'config')) {
        $schema['config'][$key] = $this->buildFieldSchema($key, $default, $fieldType, 'config', $enhancements);
      }
    }

    // Add virtual fields from enhancements (fields not in defaults but useful).
    foreach (['storage', 'config'] as $context) {
      if (!empty($enhancements[$context])) {
        foreach ($enhancements[$context] as $key => $enhancement) {
          if (!isset($schema[$context][$key]) && !empty($enhancement['add_field'])) {
            $schema[$context][$key] = $enhancement;
            unset($schema[$context][$key]['add_field']);
          }
        }
      }
    }

    // Allow modules to alter the schema.
    $this->moduleHandler->alter('eb_field_settings_schema', $schema, $fieldType);

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public function getAllFormSchemas(): array {
    $schemas = [];
    $definitions = $this->fieldTypeManager->getDefinitions();

    foreach (array_keys($definitions) as $fieldType) {
      $schema = $this->getFormSchema($fieldType);
      if (!empty($schema['storage']) || !empty($schema['config'])) {
        $schemas[$fieldType] = $schema;
      }
    }

    return $schemas;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultSettings(string $fieldType): ?array {
    $formSchema = $this->getFormSchema($fieldType);

    if (empty($formSchema)) {
      return NULL;
    }

    $storageSettings = [];
    $configSettings = [];

    if (!empty($formSchema['storage'])) {
      foreach ($formSchema['storage'] as $key => $fieldDef) {
        if (array_key_exists('default', $fieldDef)) {
          $storageSettings[$key] = $fieldDef['default'];
        }
      }
    }

    if (!empty($formSchema['config'])) {
      foreach ($formSchema['config'] as $key => $fieldDef) {
        if (array_key_exists('default', $fieldDef)) {
          $configSettings[$key] = $fieldDef['default'];
        }
      }
    }

    if (empty($storageSettings) && empty($configSettings)) {
      return NULL;
    }

    return [
      'storage' => $storageSettings,
      'config' => $configSettings,
    ];
  }

  /**
   * Determine if a field setting should be exposed in the UI.
   *
   * Uses a blocklist approach - blocks known internal settings but allows
   * everything else. This ensures contrib field types work out of the box.
   *
   * @param string $key
   *   The setting key.
   * @param mixed $default
   *   The default value.
   * @param string $fieldType
   *   The field type ID.
   * @param string $context
   *   Either 'storage' or 'config'.
   *
   * @return bool
   *   TRUE if the setting should be shown to users.
   */
  protected function shouldExpose(string $key, mixed $default, string $fieldType, string $context): bool {
    // Blocklist of internal settings that should never be shown.
    $blocklist = [
      // Handler/plugin settings (complex nested structures).
      'handler',
      'handler_settings',
      // Default values for files/images (complex nested structures).
      'default_image',
      // Display settings (internal to field API).
      'display_field',
      'display_default',
      // Callback functions (not user-configurable).
      'allowed_values_function',
      // File directory tokens (advanced, confusing for users).
      'file_directory',
    ];

    // Check blocklist.
    if (in_array($key, $blocklist, TRUE)) {
      return FALSE;
    }

    // Block target_type for file/image (always 'file', not configurable).
    if ($key === 'target_type' && in_array($fieldType, ['file', 'image'], TRUE)) {
      return FALSE;
    }

    // Skip complex nested arrays/objects (likely internal configuration).
    if (is_array($default) && !empty($default)) {
      foreach ($default as $value) {
        if (is_array($value) || is_object($value)) {
          return FALSE;
        }
      }
    }

    return TRUE;
  }

  /**
   * Build the schema definition for a single field setting.
   *
   * @param string $key
   *   The setting key.
   * @param mixed $default
   *   The default value from Drupal's field type.
   * @param string $fieldType
   *   The field type ID.
   * @param string $context
   *   Either 'storage' or 'config'.
   * @param array<string, array<string, array<string, mixed>>> $enhancements
   *   Enhancements for this field type.
   *
   * @return array<string, mixed>
   *   The schema definition for this setting.
   */
  protected function buildFieldSchema(string $key, mixed $default, string $fieldType, string $context, array $enhancements): array {
    // Normalize default value.
    $normalizedDefault = $this->normalizeValue($default);

    // Start with inferred type and humanized label.
    $schema = [
      'type' => $this->inferType($key, $normalizedDefault),
      'label' => $this->humanizeKey($key),
      'default' => $normalizedDefault,
    ];

    // Apply enhancements if available (overrides inferred values).
    if (isset($enhancements[$context][$key])) {
      $schema = array_merge($schema, $enhancements[$context][$key]);
    }

    return $schema;
  }

  /**
   * Infer form field type from value type and key name.
   *
   * @param string $key
   *   The setting key name.
   * @param mixed $default
   *   The default value.
   *
   * @return string
   *   The form field type: checkbox, number, select, checkboxes, keyvalue, text.
   */
  protected function inferType(string $key, mixed $default): string {
    if (is_bool($default)) {
      return 'checkbox';
    }
    if (is_int($default) || is_float($default)) {
      return 'number';
    }
    if (is_array($default)) {
      return 'keyvalue';
    }

    $selectFields = ['target_type', 'handler', 'datetime_type', 'uri_scheme', 'link_type'];
    if (in_array($key, $selectFields, TRUE)) {
      return 'select';
    }

    return 'text';
  }

  /**
   * Normalize a default value for JSON serialization.
   *
   * @param mixed $value
   *   The value to normalize.
   *
   * @return mixed
   *   The normalized value.
   */
  protected function normalizeValue(mixed $value): mixed {
    if ($value instanceof TranslatableMarkup) {
      return (string) $value;
    }

    if (is_array($value)) {
      return array_map([$this, 'normalizeValue'], $value);
    }

    return $value;
  }

  /**
   * Convert snake_case key to human-readable label.
   *
   * @param string $key
   *   The setting key in snake_case.
   *
   * @return string
   *   Human-readable label.
   */
  protected function humanizeKey(string $key): string {
    return ucfirst(str_replace('_', ' ', $key));
  }

  /**
   * Get enhancements (labels, descriptions, options) for field type settings.
   *
   * Provides human-friendly overrides for settings from Drupal's field type
   * classes. Contrib modules can use hook_alter to add their own.
   *
   * @param string $fieldType
   *   The field type ID.
   *
   * @return array<string, array<string, array<string, mixed>>>
   *   Enhancements keyed by context (storage/config) then setting key.
   */
  protected function getEnhancements(string $fieldType): array {
    // Common enhancements that apply to any field type with these settings.
    $common = [
      'storage' => [
        'max_length' => [
          'label' => 'Maximum length',
          'description' => 'Maximum number of characters.',
          'min' => 1,
          'max' => 255,
        ],
        'precision' => [
          'label' => 'Precision',
          'description' => 'Total number of digits.',
          'min' => 1,
          'max' => 32,
        ],
        'scale' => [
          'label' => 'Scale',
          'description' => 'Number of decimal places.',
          'min' => 0,
          'max' => 10,
        ],
        'target_type' => [
          'type' => 'select',
          'label' => 'Target entity type',
          'description' => 'The type of entity to reference.',
          'options_source' => 'entityTypes',
          'required' => TRUE,
        ],
        'uri_scheme' => [
          'type' => 'select',
          'label' => 'Upload destination',
          'description' => 'File storage location.',
          'options' => [
            'public' => 'Public files',
            'private' => 'Private files',
          ],
        ],
        'datetime_type' => [
          'type' => 'select',
          'label' => 'Date type',
          'description' => 'Whether to store date only or date and time.',
          'options' => [
            'datetime' => 'Date and time',
            'date' => 'Date only',
          ],
        ],
        'allowed_values' => [
          'type' => 'keyvalue',
          'label' => 'Allowed values',
          'description' => 'Enter key|label pairs for the options.',
        ],
        'is_ascii' => [
          'type' => 'checkbox',
          'label' => 'ASCII only',
          'description' => 'Restrict to ASCII characters only.',
        ],
        'case_sensitive' => [
          'type' => 'checkbox',
          'label' => 'Case sensitive',
          'description' => 'Make values case sensitive.',
        ],
      ],
      'config' => [
        'file_extensions' => [
          'label' => 'Allowed file extensions',
          'description' => 'Space-separated list of allowed extensions.',
        ],
        'max_filesize' => [
          'label' => 'Maximum upload size',
          'description' => 'E.g., 2 MB or 512 KB. Leave empty for no limit.',
        ],
        'max_resolution' => [
          'label' => 'Maximum image dimensions',
          'description' => 'E.g., 1920x1080. Leave empty for no limit.',
        ],
        'min_resolution' => [
          'label' => 'Minimum image dimensions',
          'description' => 'E.g., 100x100. Leave empty for no limit.',
        ],
        'alt_field' => [
          'type' => 'checkbox',
          'label' => 'Enable Alt text',
          'description' => 'Alternative text for accessibility.',
        ],
        'alt_field_required' => [
          'type' => 'checkbox',
          'label' => 'Require Alt text',
          'description' => 'Make alternative text mandatory.',
        ],
        'title_field' => [
          'type' => 'checkbox',
          'label' => 'Enable Title text',
          'description' => 'Title attribute for tooltips.',
        ],
        'title_field_required' => [
          'type' => 'checkbox',
          'label' => 'Require Title text',
          'description' => 'Make title text mandatory.',
        ],
        'min' => [
          'label' => 'Minimum value',
          'description' => 'Leave empty for no limit.',
        ],
        'max' => [
          'label' => 'Maximum value',
          'description' => 'Leave empty for no limit.',
        ],
        'on_label' => [
          'label' => 'On label',
          'description' => 'Text shown when value is true/yes.',
        ],
        'off_label' => [
          'label' => 'Off label',
          'description' => 'Text shown when value is false/no.',
        ],
        'allowed_formats' => [
          'type' => 'checkboxes',
          'label' => 'Allowed text formats',
          'description' => 'Select which text formats are allowed. Leave empty to allow all formats.',
          'options_source' => 'textFormats',
        ],
      ],
    ];

    $enhancements = $common;

    // Field-type-specific enhancements.
    $specific = [
      'entity_reference' => [
        'config' => [
          // Virtual field - not in defaults but useful to show.
          // Nested inside handler_settings in Drupal's field config.
          'target_bundles' => [
            'add_field' => TRUE,
            'type' => 'checkboxes',
            'label' => 'Allowed bundles',
            'description' => 'Leave empty to allow all bundles.',
            'options_source' => 'bundlesForTargetType',
            'default' => [],
            'nested_path' => 'handler_settings.target_bundles',
          ],
        ],
      ],
      'text_with_summary' => [
        'config' => [
          'display_summary' => [
            'type' => 'checkbox',
            'label' => 'Display summary',
            'description' => 'Show a separate summary field.',
          ],
        ],
      ],
      'link' => [
        'config' => [
          'link_type' => [
            'type' => 'select',
            'label' => 'Allowed link type',
            'description' => 'What type of links are allowed.',
            'options' => [
              1 => 'Internal links only',
              16 => 'External links only',
              17 => 'Both internal and external',
            ],
          ],
          'title' => [
            'type' => 'select',
            'label' => 'Link title',
            'description' => 'Whether to allow or require link title.',
            'options' => [
              0 => 'Disabled',
              1 => 'Optional',
              2 => 'Required',
            ],
          ],
        ],
      ],
    ];

    // Merge field-type-specific enhancements.
    if (isset($specific[$fieldType])) {
      foreach (['storage', 'config'] as $context) {
        if (isset($specific[$fieldType][$context])) {
          $enhancements[$context] = array_merge(
            $enhancements[$context],
            $specific[$fieldType][$context]
          );
        }
      }
    }

    return $enhancements;
  }

}
