<?php

namespace Drupal\Tests\eb_aggrid\Functional;

use Psr\Http\Message\ResponseInterface;
use Behat\Mink\Driver\BrowserKitDriver;
use GuzzleHttp\Client;
use GuzzleHttp\Cookie\CookieJar;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Entity Builder AG-Grid API endpoints.
 *
 * @group eb_aggrid
 */
class ApiEndpointsTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['eb', 'eb_ui', 'eb_aggrid', 'node', 'field', 'user'];

  /**
   * A user with permission to use entity builder.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'import entity architecture',
      'administer entity builder',
    ]);
  }

  /**
   * Tests CSRF protection - requests without X-Requested-With are rejected.
   *
   * @covers \Drupal\eb_ui\Controller\EbUiApiController::validateAjaxRequest
   */
  public function testCsrfProtectionValidate(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/validate');

    // Request WITHOUT X-Requested-With header should be rejected.
    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
      ],
      'body' => '{"test": "data"}',
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    $this->assertEquals(403, $response->getStatusCode());
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertFalse($data['valid']);
    $this->assertStringContainsString('X-Requested-With', $data['errors'][0]);
  }

  /**
   * Tests CSRF protection - requests with X-Requested-With but no CSRF token.
   *
   * @covers \Drupal\eb_ui\Controller\EbUiApiController::validateAjaxRequest
   */
  public function testCsrfProtectionValidateWithHeaderNoCsrf(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/validate');

    // Request WITH X-Requested-With but WITHOUT CSRF token should be rejected.
    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-Requested-With' => 'XMLHttpRequest',
      ],
      'body' => '',
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    // Should get 403 for missing CSRF token.
    $this->assertEquals(403, $response->getStatusCode());
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertStringContainsString('CSRF token', $data['errors'][0] ?? '');
  }

  /**
   * Tests CSRF protection - requests with valid headers are accepted.
   *
   * @covers \Drupal\eb_ui\Controller\EbUiApiController::validateAjaxRequest
   */
  public function testCsrfProtectionValidateWithAllHeaders(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/validate');

    // Request WITH X-Requested-With AND CSRF token should be accepted.
    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'body' => '',
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    // Should get 400 for empty content, not 403 for CSRF.
    $this->assertEquals(400, $response->getStatusCode());
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertStringNotContainsString('CSRF', $data['errors'][0] ?? '');
  }

  /**
   * Tests CSRF protection on preview endpoint.
   *
   * @covers \Drupal\eb_ui\Controller\EbUiApiController::validateAjaxRequest
   */
  public function testCsrfProtectionPreview(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/preview');

    // Request WITHOUT X-Requested-With header should be rejected.
    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
      ],
      'body' => '{"test": "data"}',
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    $this->assertEquals(403, $response->getStatusCode());
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertFalse($data['success']);
  }

  /**
   * Tests CSRF protection on bundles endpoint.
   *
   * @covers \Drupal\eb_ui\Controller\EbUiApiController::getBundles
   */
  public function testCsrfProtectionBundles(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/bundles/node');

    // GET requests don't require CSRF token, just permission check.
    $response = $client->get($url, [
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    // Should succeed since user has permission.
    $this->assertEquals(200, $response->getStatusCode());
  }

  /**
   * Tests anonymous users are rejected by permission check.
   *
   * @covers \Drupal\eb_ui\Controller\EbUiApiController::validateAjaxRequest
   */
  public function testCsrfProtectionAnonymous(): void {
    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/validate');

    // Anonymous request should be rejected by Drupal's access system.
    // The route has _permission: 'import entity architecture' requirement,
    // so anonymous users are rejected before the controller even runs.
    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-Requested-With' => 'XMLHttpRequest',
      ],
      'body' => '{}',
      'http_errors' => FALSE,
    ]);

    // Anonymous users get 403 from Drupal's access system.
    $this->assertEquals(403, $response->getStatusCode());
  }

  /**
   * Tests access to API validation endpoint.
   */
  public function testApiValidationAccess(): void {
    // Anonymous users should not have access.
    $response = $this->drupalPostJsonWithXhr('/eb/api/validate', []);
    $this->assertEquals(403, $response->getStatusCode());

    // Admin user should have access.
    $this->drupalLogin($this->adminUser);
    $response = $this->drupalPostJsonWithXhr('/eb/api/validate', []);
    // Will return 400 because no content provided, but that's expected.
    $this->assertNotEquals(403, $response->getStatusCode());
  }

  /**
   * Tests API validation with empty content.
   */
  public function testApiValidationEmptyContent(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/validate');

    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    // Verify status code - should be 400 for empty content.
    $statusCode = $response->getStatusCode();
    // Reset stream position and get body content.
    $response->getBody()->rewind();
    $body = $response->getBody()->getContents();

    // If response is not what we expect, provide useful debug info.
    if ($statusCode !== 400) {
      $this->fail("Expected status 400, got $statusCode. Body: " . substr($body, 0, 500));
    }

    $data = json_decode($body, TRUE);
    $this->assertNotNull($data, 'Response should be valid JSON. Status: ' . $statusCode . ', Body: ' . substr($body, 0, 500));
    $this->assertFalse($data['valid']);
    $this->assertArrayHasKey('errors', $data);
  }

  /**
   * Tests API validation with valid YAML.
   */
  public function testApiValidationValidYaml(): void {
    $this->drupalLogin($this->adminUser);

    $yaml_content = <<<YAML
- operation: create_bundle
  entity_type: node
  bundle: test
  label: Test
YAML;

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/validate');

    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'text/plain',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'body' => $yaml_content,
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    // Note: Actual validation result depends on operation plugins.
    // At minimum, the endpoint should process the request.
    $this->assertContains($response->getStatusCode(), [200, 400]);

    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertIsArray($data);
  }

  /**
   * Tests API preview endpoint access.
   */
  public function testApiPreviewAccess(): void {
    // Anonymous users should not have access.
    $response = $this->drupalPostJsonWithXhr('/eb/api/preview', []);
    $this->assertEquals(403, $response->getStatusCode());

    // Admin user should have access.
    $this->drupalLogin($this->adminUser);
    $response = $this->drupalPostJsonWithXhr('/eb/api/preview', []);
    $this->assertNotEquals(403, $response->getStatusCode());
  }

  /**
   * Tests API preview with empty content.
   */
  public function testApiPreviewEmptyContent(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/preview');

    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    $this->assertEquals(400, $response->getStatusCode());

    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertFalse($data['success']);
  }

  /**
   * Tests API preview with valid content.
   */
  public function testApiPreviewValidContent(): void {
    $this->drupalLogin($this->adminUser);

    $yaml_content = <<<YAML
- operation: create_bundle
  entity_type: node
  bundle_id: test_preview
  label: Test Preview
YAML;

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/preview');

    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'text/plain',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'body' => $yaml_content,
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    // The endpoint should process the request.
    $this->assertContains($response->getStatusCode(), [200, 400]);

    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertIsArray($data);
  }

  /**
   * Tests bundles endpoint with valid request.
   */
  public function testBundlesEndpoint(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/bundles/node');

    $response = $client->get($url, [
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    $this->assertEquals(200, $response->getStatusCode());
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertTrue($data['success']);
    $this->assertArrayHasKey('bundles', $data);
  }

  /**
   * Tests JSON depth validation prevents DoS attacks.
   */
  public function testJsonDepthValidation(): void {
    $this->drupalLogin($this->adminUser);

    // Create deeply nested JSON (beyond MAX_JSON_DEPTH of 10).
    $nested = ['value' => 'test'];
    for ($i = 0; $i < 15; $i++) {
      $nested = ['nested' => $nested];
    }

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/validate');

    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'body' => json_encode($nested),
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    $this->assertEquals(400, $response->getStatusCode());
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertFalse($data['valid']);
    $this->assertStringContainsString('nesting depth', $data['errors'][0]);
  }

  /**
   * Tests content size limit.
   */
  public function testContentSizeLimit(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/validate');

    // Create a large content payload (over 5MB).
    $largeContent = str_repeat('x', 5242881);

    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'text/plain',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
        'Content-Length' => strlen($largeContent),
      ],
      'body' => $largeContent,
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    $this->assertEquals(413, $response->getStatusCode());
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertFalse($data['valid']);
    $this->assertStringContainsString('maximum allowed size', $data['errors'][0]);
  }

  /**
   * Tests plugin settings endpoint for widget.
   *
   * @covers \Drupal\eb_aggrid\Controller\EbAggridApiController::getPluginSettingsForm
   */
  public function testPluginSettingsEndpointWidget(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/plugin-settings/widget/string_textfield');

    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'body' => json_encode(['field_type' => 'string']),
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    // The endpoint should process the request.
    $this->assertContains($response->getStatusCode(), [200, 400]);
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertIsArray($data);
  }

  /**
   * Tests plugin settings endpoint for formatter.
   *
   * @covers \Drupal\eb_aggrid\Controller\EbAggridApiController::getPluginSettingsForm
   */
  public function testPluginSettingsEndpointFormatter(): void {
    $this->drupalLogin($this->adminUser);

    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/eb/api/plugin-settings/formatter/string');

    $response = $client->post($url, [
      'headers' => [
        'Content-Type' => 'application/json',
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'body' => json_encode(['field_type' => 'string']),
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    // The endpoint should process the request.
    $this->assertContains($response->getStatusCode(), [200, 400]);
    $response->getBody()->rewind();
    $data = json_decode($response->getBody()->getContents(), TRUE);
    $this->assertIsArray($data);
  }

  /**
   * Helper to get session cookies for authenticated requests.
   *
   * @return \GuzzleHttp\Cookie\CookieJar
   *   The cookie jar with session cookies.
   */
  protected function getSessionCookies(): CookieJar {
    $cookies = [];
    $driver = $this->getSession()->getDriver();
    assert($driver instanceof BrowserKitDriver);
    foreach ($driver->getClient()->getCookieJar()->all() as $cookie) {
      $cookies[$cookie->getName()] = $cookie->getValue();
    }
    // Create a CookieJar from the cookie array.
    return CookieJar::fromArray($cookies, parse_url($this->baseUrl, PHP_URL_HOST));
  }

  /**
   * Helper to get CSRF token from the session/token endpoint.
   *
   * @return string
   *   The CSRF token.
   */
  protected function getCsrfToken(): string {
    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl('/session/token');

    $response = $client->get($url, [
      'cookies' => $this->getSessionCookies(),
    ]);

    $response->getBody()->rewind();
    return $response->getBody()->getContents();
  }

  /**
   * Helper to perform JSON POST request without X-Requested-With header.
   *
   * @param string $path
   *   The path to post to.
   * @param array<mixed> $data
   *   The data to post.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response.
   */
  protected function drupalPostJson(string $path, array $data): ResponseInterface {
    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl($path);

    $response = $client->post($url, [
      'json' => $data,
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    return $response;
  }

  /**
   * Helper to perform JSON POST with X-Requested-With header.
   *
   * This simulates a proper AJAX request with CSRF protection.
   *
   * @param string $path
   *   The path to post to.
   * @param array<mixed> $data
   *   The data to post.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response.
   */
  protected function drupalPostJsonWithXhr(string $path, array $data): ResponseInterface {
    $client = $this->getHttpClient();
    assert($client instanceof Client);
    $url = $this->buildUrl($path);

    $response = $client->post($url, [
      'json' => $data,
      'headers' => [
        'X-Requested-With' => 'XMLHttpRequest',
        'X-CSRF-Token' => $this->getCsrfToken(),
      ],
      'cookies' => $this->getSessionCookies(),
      'http_errors' => FALSE,
    ]);

    return $response;
  }

}
