<?php

namespace Drupal\Tests\eb_aggrid\Kernel\Controller;

use Drupal\Core\Access\CsrfRequestHeaderAccessCheck;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\eb_aggrid\Controller\EbAggridApiController;
use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests the EbAggridApiController.
 *
 * @group eb_aggrid
 * @coversDefaultClass \Drupal\eb_aggrid\Controller\EbAggridApiController
 */
class EbAggridApiControllerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'node',
    'eb',
    'eb_ui',
    'eb_aggrid',
  ];

  /**
   * The controller under test.
   */
  protected EbAggridApiController $controller;

  /**
   * The CSRF token generator.
   */
  protected CsrfTokenGenerator $csrfTokenGenerator;

  /**
   * The authenticated user.
   */
  protected User $authenticatedUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installConfig(['system', 'user', 'eb', 'eb_ui', 'eb_aggrid']);
    $this->installSchema('user', ['users_data']);

    // Create controller.
    $this->controller = EbAggridApiController::create($this->container);
    $this->csrfTokenGenerator = $this->container->get('csrf_token');

    // Create and set authenticated user.
    $this->authenticatedUser = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $this->authenticatedUser->save();
    $this->container->get('current_user')->setAccount($this->authenticatedUser);
  }

  /**
   * Tests that requests without X-Requested-With header are rejected.
   *
   * @covers ::validateAjaxRequest
   */
  public function testRejectsMissingAjaxHeader(): void {
    $request = Request::create('/eb/api/plugin-settings/widget/string_textfield', 'POST', [], [], [], [], '{}');
    $response = $this->controller->getPluginSettingsForm($request, 'widget', 'string_textfield');

    $this->assertEquals(403, $response->getStatusCode());
    $data = json_decode($response->getContent(), TRUE);
    $this->assertFalse($data['success']);
    $this->assertStringContainsString('Missing X-Requested-With header', $data['errors'][0]);
  }

  /**
   * Tests that requests without CSRF token are rejected.
   *
   * @covers ::validateAjaxRequest
   */
  public function testRejectsMissingCsrfToken(): void {
    $request = Request::create('/eb/api/plugin-settings/widget/string_textfield', 'POST', [], [], [], [], '{}');
    $request->headers->set('X-Requested-With', 'XMLHttpRequest');

    $response = $this->controller->getPluginSettingsForm($request, 'widget', 'string_textfield');

    $this->assertEquals(403, $response->getStatusCode());
    $data = json_decode($response->getContent(), TRUE);
    $this->assertFalse($data['success']);
    $this->assertStringContainsString('CSRF token validation failed', $data['errors'][0]);
  }

  /**
   * Tests that requests with invalid CSRF token are rejected.
   *
   * @covers ::validateAjaxRequest
   */
  public function testRejectsInvalidCsrfToken(): void {
    $request = Request::create('/eb/api/plugin-settings/widget/string_textfield', 'POST', [], [], [], [], '{}');
    $request->headers->set('X-Requested-With', 'XMLHttpRequest');
    $request->headers->set('X-CSRF-Token', 'invalid_token');

    $response = $this->controller->getPluginSettingsForm($request, 'widget', 'string_textfield');

    $this->assertEquals(403, $response->getStatusCode());
    $data = json_decode($response->getContent(), TRUE);
    $this->assertFalse($data['success']);
    $this->assertStringContainsString('CSRF token validation failed', $data['errors'][0]);
  }

  /**
   * Tests that requests with valid CSRF token are accepted.
   *
   * @covers ::validateAjaxRequest
   */
  public function testAcceptsValidCsrfToken(): void {
    $csrfToken = $this->csrfTokenGenerator->get(CsrfRequestHeaderAccessCheck::TOKEN_KEY);

    $request = Request::create('/eb/api/plugin-settings/widget/string_textfield', 'POST', [], [], [], [], '{}');
    $request->headers->set('X-Requested-With', 'XMLHttpRequest');
    $request->headers->set('X-CSRF-Token', $csrfToken);

    $response = $this->controller->getPluginSettingsForm($request, 'widget', 'string_textfield');

    // Should not return 403 - may return other status for missing data.
    $this->assertNotEquals(403, $response->getStatusCode());
  }

  /**
   * Tests content size limit validation.
   *
   * @covers ::validateAjaxRequest
   */
  public function testRejectsExcessiveContentSize(): void {
    $csrfToken = $this->csrfTokenGenerator->get(CsrfRequestHeaderAccessCheck::TOKEN_KEY);

    $request = Request::create('/eb/api/plugin-settings/widget/string_textfield', 'POST', [], [], [], [], '{}');
    $request->headers->set('X-Requested-With', 'XMLHttpRequest');
    $request->headers->set('X-CSRF-Token', $csrfToken);
    // Set content length to exceed MAX_CONTENT_SIZE (5MB).
    $request->headers->set('Content-Length', '6000000');

    $response = $this->controller->getPluginSettingsForm($request, 'widget', 'string_textfield');

    $this->assertEquals(413, $response->getStatusCode());
    $data = json_decode($response->getContent(), TRUE);
    $this->assertFalse($data['success']);
    $this->assertStringContainsString('exceeds maximum allowed size', $data['errors'][0]);
  }

  /**
   * Tests anonymous user rejection.
   *
   * @covers ::validateAjaxRequest
   */
  public function testRejectsAnonymousUser(): void {
    // Set anonymous user.
    $anonymousUser = User::getAnonymousUser();
    $this->container->get('current_user')->setAccount($anonymousUser);

    $request = Request::create('/eb/api/plugin-settings/widget/string_textfield', 'POST', [], [], [], [], '{}');
    $request->headers->set('X-Requested-With', 'XMLHttpRequest');

    $response = $this->controller->getPluginSettingsForm($request, 'widget', 'string_textfield');

    $this->assertEquals(403, $response->getStatusCode());
    $data = json_decode($response->getContent(), TRUE);
    $this->assertFalse($data['success']);
    $this->assertStringContainsString('Authentication required', $data['errors'][0]);
  }

}
