<?php

namespace Drupal\Tests\eb_aggrid\Unit\Form;

use Drupal\Component\Utility\Xss;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the sanitization logic used in EbAggridForm.
 *
 * @group eb_aggrid
 * @coversDefaultClass \Drupal\eb_aggrid\Form\EbAggridForm
 */
class SanitizationTest extends UnitTestCase {

  /**
   * Tests that script tags are removed from strings.
   *
   * Xss::filter() removes the script tags but keeps the text content,
   * making it safe for display.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizeRemovesScriptTags(): void {
    $malicious = '<script>alert("xss")</script>safe';
    $result = Xss::filter($malicious);
    // Tags are removed, content remains.
    $this->assertStringNotContainsString('<script>', $result);
    $this->assertStringNotContainsString('</script>', $result);
    $this->assertStringContainsString('safe', $result);
  }

  /**
   * Tests that event handlers are removed from HTML.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizeRemovesEventHandlers(): void {
    $malicious = '<div onclick="alert(1)">content</div>';
    $result = Xss::filter($malicious);
    $this->assertStringNotContainsString('onclick', $result);
    $this->assertStringContainsString('content', $result);
  }

  /**
   * Tests that javascript: protocol links are sanitized.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizeRemovesJavascriptProtocol(): void {
    $malicious = '<a href="javascript:alert(1)">click</a>';
    $result = Xss::filter($malicious);
    $this->assertStringNotContainsString('javascript:', $result);
  }

  /**
   * Tests that safe text is preserved.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizePreservesSafeText(): void {
    $safe = 'This is a normal field label';
    $result = Xss::filter($safe);
    $this->assertEquals($safe, $result);
  }

  /**
   * Tests that safe HTML is preserved.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizePreservesSafeHtml(): void {
    $safe = '<em>emphasized</em> and <strong>bold</strong>';
    $result = Xss::filter($safe);
    $this->assertStringContainsString('<em>', $result);
    $this->assertStringContainsString('<strong>', $result);
  }

  /**
   * Tests sanitizing grid data with nested arrays.
   *
   * Xss::filter() removes dangerous tags but keeps the text content.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizeGridDataNested(): void {
    $data = [
      [
        'field_name' => 'field_<script>xss</script>test',
        'label' => 'Test <em>Label</em>',
        'settings' => [
          'allowed_values' => [
            '<script>bad</script>Value 1',
            'Value 2',
          ],
        ],
      ],
    ];

    $sanitized = $this->sanitizeGridData($data);

    // Script tags removed but text content preserved.
    $this->assertStringNotContainsString('<script>', $sanitized[0]['field_name']);
    $this->assertStringContainsString('test', $sanitized[0]['field_name']);
    // Em is allowed in Xss::filter().
    $this->assertStringContainsString('<em>', $sanitized[0]['label']);
    // Script tags removed from nested values.
    $this->assertStringNotContainsString('<script>', $sanitized[0]['settings']['allowed_values'][0]);
    $this->assertStringContainsString('Value 1', $sanitized[0]['settings']['allowed_values'][0]);
    $this->assertEquals('Value 2', $sanitized[0]['settings']['allowed_values'][1]);
  }

  /**
   * Tests that non-string values are preserved.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizeGridDataPreservesTypes(): void {
    $data = [
      [
        'weight' => 10,
        'required' => TRUE,
        'cardinality' => -1,
        'label' => 'Test',
        'nullable' => NULL,
      ],
    ];

    $sanitized = $this->sanitizeGridData($data);

    $this->assertSame(10, $sanitized[0]['weight']);
    $this->assertTrue($sanitized[0]['required']);
    $this->assertSame(-1, $sanitized[0]['cardinality']);
    $this->assertSame('Test', $sanitized[0]['label']);
    $this->assertNull($sanitized[0]['nullable']);
  }

  /**
   * Tests sanitization with deeply nested data.
   *
   * Xss::filter() removes dangerous tags but keeps the text content.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizeGridDataDeepNesting(): void {
    // Create data with 8 levels of nesting (within the 10 limit).
    $data = [
      'level1' => [
        'level2' => [
          'level3' => [
            'level4' => [
              'level5' => [
                'level6' => [
                  'level7' => [
                    'level8' => '<script>xss</script>value',
                  ],
                ],
              ],
            ],
          ],
        ],
      ],
    ];

    $sanitized = $this->sanitizeGridData($data);

    $nested = $sanitized['level1']['level2']['level3']['level4']['level5']['level6']['level7']['level8'];
    // Script tags removed but text content preserved.
    $this->assertStringNotContainsString('<script>', $nested);
    $this->assertStringContainsString('value', $nested);
  }

  /**
   * Tests sanitization prevents excessive nesting.
   *
   * @covers ::sanitizeGridData
   */
  public function testSanitizeGridDataExcessiveNesting(): void {
    // Create data with 12 levels of nesting (beyond the 10 limit).
    $data = [];
    $current = &$data;
    for ($i = 1; $i <= 12; $i++) {
      $current["level$i"] = [];
      $current = &$current["level$i"];
    }
    $current['value'] = 'test';

    $sanitized = $this->sanitizeGridData($data);

    // At depth 10, the function returns empty array for further nesting.
    // Navigate down to where truncation should occur.
    $nested = $sanitized;
    for ($i = 1; $i <= 10; $i++) {
      if (!isset($nested["level$i"])) {
        // We expect truncation at some point.
        break;
      }
      $nested = $nested["level$i"];
    }
    // The data beyond depth 10 should be truncated.
    $this->assertIsArray($nested);
  }

  /**
   * Tests sanitization of various XSS vectors.
   *
   * @covers ::sanitizeGridData
   * @dataProvider xssVectorProvider
   */
  public function testSanitizeXssVectors(string $input, string $forbidden): void {
    $result = Xss::filter($input);
    $this->assertStringNotContainsString($forbidden, strtolower($result));
  }

  /**
   * Provides XSS attack vectors for testing.
   *
   * @return array<array{string, string}>
   *   Test data.
   */
  public static function xssVectorProvider(): array {
    return [
      'script tag' => ['<script>alert(1)</script>', 'script'],
      'img onerror' => ['<img src=x onerror=alert(1)>', 'onerror'],
      'svg onload' => ['<svg onload=alert(1)>', 'onload'],
      'iframe' => ['<iframe src="evil.com"></iframe>', 'iframe'],
      'style tag' => ['<style>body{background:url(evil)}</style>', 'style'],
      'object tag' => ['<object data="evil.swf"></object>', 'object'],
      'embed tag' => ['<embed src="evil.swf">', 'embed'],
      'base tag' => ['<base href="http://evil.com/">', 'base'],
    ];
  }

  /**
   * Sanitize grid data helper method.
   *
   * This replicates the logic from EbAggridForm::sanitizeGridData().
   *
   * @param array<mixed> $data
   *   The data to sanitize.
   * @param int $depth
   *   Current recursion depth.
   *
   * @return array<mixed>
   *   The sanitized data.
   */
  protected function sanitizeGridData(array $data, int $depth = 0): array {
    // Prevent infinite recursion.
    if ($depth > 10) {
      return [];
    }

    $sanitized = [];
    foreach ($data as $key => $value) {
      if (is_string($value)) {
        // Sanitize string values.
        $sanitized[$key] = Xss::filter($value);
      }
      elseif (is_array($value)) {
        // Recursively sanitize nested arrays.
        $sanitized[$key] = $this->sanitizeGridData($value, $depth + 1);
      }
      else {
        // Keep other types (int, bool, null) as-is.
        $sanitized[$key] = $value;
      }
    }

    return $sanitized;
  }

}
