<?php

namespace Drupal\eb_auto_entitylabel\Plugin\EbExtension;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eb\Attribute\EbExtension;
use Drupal\eb\PluginBase\EbExtensionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides automatic entity label support for Entity Builder.
 *
 * Creates auto_entitylabel.settings config objects.
 *
 * Example YAML (auto_entitylabel is columns on bundle_definitions):
 *
 * bundle_definitions:
 *   - entity_type: node
 *     bundle_id: article
 *     label: Article
 *     auto_entitylabel_status: enabled
 *     auto_entitylabel_pattern: "[node:field_date] - [node:field_title]"
 *     auto_entitylabel_escape: false
 *     auto_entitylabel_preserve_titles: false
 */
#[EbExtension(
  id: 'auto_entitylabel',
  label: new TranslatableMarkup('Auto Entity Label'),
  description: new TranslatableMarkup('Provides automatic label pattern configuration via YAML'),
  yaml_keys: ['auto_entitylabel_status', 'auto_entitylabel_pattern'],
  operations: ['configure_auto_entitylabel'],
  module_dependencies: ['auto_entitylabel'],
)]
class AutoEntityLabelExtension extends EbExtensionBase {

  /**
   * Status value mapping from YAML strings to integer constants.
   */
  protected const STATUS_MAP = [
    'disabled' => 0,
    'enabled' => 1,
    'optional' => 2,
    'prefilled' => 3,
  ];

  /**
   * New content behavior mapping from YAML strings to integer constants.
   */
  protected const BEHAVIOR_MAP = [
    'before_save' => 0,
    'after_save' => 1,
  ];

  /**
   * The config factory service.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOperations(array $data): array {
    $operations = [];

    // Auto entity label config is in bundle_definitions columns.
    if (empty($data['bundle_definitions']) || !is_array($data['bundle_definitions'])) {
      return $operations;
    }

    foreach ($data['bundle_definitions'] as $bundle) {
      if (!is_array($bundle) || !isset($bundle['auto_entitylabel_status'])) {
        continue;
      }

      $entity_type = $bundle['entity_type'] ?? '';
      $bundle_id = $bundle['bundle_id'] ?? $bundle['id'] ?? '';

      if (!$entity_type || !$bundle_id) {
        continue;
      }

      // Map status string to integer.
      $status_string = $bundle['auto_entitylabel_status'] ?? 'disabled';
      $status = self::STATUS_MAP[$status_string] ?? 0;

      // Map behavior string to integer.
      $behavior_string = $bundle['auto_entitylabel_behavior'] ?? 'before_save';
      $behavior = self::BEHAVIOR_MAP[$behavior_string] ?? 0;

      $operations[] = [
        'operation' => 'configure_auto_entitylabel',
        'entity_type' => $entity_type,
        'bundle' => $bundle_id,
        'status' => $status,
        'pattern' => $bundle['auto_entitylabel_pattern'] ?? '',
        'escape' => (bool) ($bundle['auto_entitylabel_escape'] ?? FALSE),
        'preserve_titles' => (bool) ($bundle['auto_entitylabel_preserve_titles'] ?? FALSE),
        'new_content_behavior' => $behavior,
      ];
    }

    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperationDependencies(array $operation, array $context): array {
    $entity_type = $operation['entity_type'] ?? '';
    $bundle = $operation['bundle'] ?? '';

    if (!$entity_type || !$bundle) {
      return [];
    }

    return ["bundle:{$entity_type}:{$bundle}"];
  }

  /**
   * {@inheritdoc}
   */
  public function detectChanges(array $operation, array $context): ?string {
    if (!$this->appliesTo($operation)) {
      return NULL;
    }

    $entity_type = $operation['entity_type'] ?? '';
    $bundle = $operation['bundle'] ?? '';
    $config = $this->configFactory->get("auto_entitylabel.settings.{$entity_type}.{$bundle}");

    if ($config->isNew()) {
      return 'configure_auto_entitylabel';
    }

    // Compare values.
    $desired_status = $operation['status'] ?? 0;
    $desired_pattern = $operation['pattern'] ?? '';
    $desired_escape = (bool) ($operation['escape'] ?? FALSE);
    $desired_preserve_titles = (bool) ($operation['preserve_titles'] ?? FALSE);
    $desired_behavior = $operation['new_content_behavior'] ?? 0;

    if ($desired_status !== $config->get('status') ||
        $desired_pattern !== $config->get('pattern') ||
        $desired_escape !== (bool) $config->get('escape') ||
        $desired_preserve_titles !== (bool) $config->get('preserve_titles') ||
        $desired_behavior !== ($config->get('new_content_behavior') ?? 0)) {
      return 'configure_auto_entitylabel';
    }

    return 'skip';
  }

}
