<?php

namespace Drupal\eb_auto_entitylabel\Plugin\EbOperation;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eb\Attribute\EbOperation;
use Drupal\eb\Exception\ExecutionException;
use Drupal\eb\Exception\RollbackException;
use Drupal\eb\PluginBase\OperationBase;
use Drupal\eb\Result\ExecutionResult;
use Drupal\eb\Result\PreviewResult;
use Drupal\eb\Result\RollbackResult;
use Drupal\eb\Result\ValidationResult;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Operation for configuring automatic entity labels.
 */
#[EbOperation(
  id: 'configure_auto_entitylabel',
  label: new TranslatableMarkup('Configure Auto Entity Label'),
  description: new TranslatableMarkup('Configures automatic label generation for a bundle'),
  operationType: 'create',
)]
class ConfigureAutoEntityLabelOperation extends OperationBase {

  /**
   * Status value labels for display.
   */
  protected const STATUS_LABELS = [
    0 => 'Disabled',
    1 => 'Enabled',
    2 => 'Optional',
    3 => 'Prefilled',
  ];

  /**
   * Constructor.
   *
   * @param array<string, mixed> $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger channel.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
    LoggerInterface $logger,
    ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityTypeManager, $logger, $configFactory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Psr\Log\LoggerInterface $logger */
    $logger = $container->get('logger.channel.eb');

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $logger,
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate(): ValidationResult {
    $result = new ValidationResult();

    // Validate required fields.
    $this->validateRequiredFields(['entity_type', 'bundle', 'status'], $result);

    if (!$result->isValid()) {
      return $result;
    }

    $status = $this->getDataValue('status');
    $pattern = $this->getDataValue('pattern', '');

    // Validate status is a valid integer.
    if (!is_numeric($status) || $status < 0 || $status > 3) {
      $result->addError(
        $this->t('Status must be 0 (disabled), 1 (enabled), 2 (optional), or 3 (prefilled).'),
        'invalid_status'
      );
    }

    // Validate pattern is present when status is not disabled.
    if ($status > 0 && empty(trim($pattern))) {
      $result->addError(
        $this->t('Pattern is required when auto entity label is enabled.'),
        'missing_pattern'
      );
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function preview(): PreviewResult {
    $entityType = $this->getDataValue('entity_type');
    $bundle = $this->getDataValue('bundle');
    $status = (int) $this->getDataValue('status');
    $pattern = $this->getDataValue('pattern', '');

    $configName = "auto_entitylabel.settings.{$entityType}.{$bundle}";

    $preview = new PreviewResult();

    // Check if config exists.
    $existingConfig = $this->configFactory->get($configName);
    $isNew = $existingConfig->isNew();

    $preview->addOperation(
      $isNew ? 'create' : 'update',
      'config',
      $configName,
      $this->t('@op auto entity label for @entity_type.@bundle', [
        '@op' => $isNew ? 'Configure' : 'Update',
        '@entity_type' => $entityType,
        '@bundle' => $bundle,
      ])
    );

    $details = [
      'Config Name' => $configName,
      'Entity Type' => $entityType,
      'Bundle' => $bundle,
      'Status' => self::STATUS_LABELS[$status] ?? 'Unknown',
    ];

    if ($status > 0) {
      $details['Pattern'] = $pattern;
    }

    $preview->addDetails($details);

    return $preview;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): ExecutionResult {
    try {
      $entityType = $this->getDataValue('entity_type');
      $bundle = $this->getDataValue('bundle');
      $status = (int) $this->getDataValue('status');
      $pattern = $this->getDataValue('pattern', '');
      $escape = (bool) $this->getDataValue('escape', FALSE);
      $preserveTitles = (bool) $this->getDataValue('preserve_titles', FALSE);
      $newContentBehavior = (int) $this->getDataValue('new_content_behavior', 0);

      $configName = "auto_entitylabel.settings.{$entityType}.{$bundle}";

      // Get or create the config.
      $config = $this->configFactory->getEditable($configName);
      $wasNew = $config->isNew();

      // Store original data for rollback.
      $originalData = NULL;
      if (!$wasNew) {
        $originalData = $config->getRawData();
      }

      // Set the configuration values.
      $config->set('status', $status);
      $config->set('pattern', $pattern);
      $config->set('escape', $escape);
      $config->set('preserve_titles', $preserveTitles);
      $config->set('new_content_behavior', $newContentBehavior);

      // Set default batch settings if not present.
      if ($config->get('save') === NULL) {
        $config->set('save', FALSE);
      }
      if ($config->get('chunk') === NULL) {
        $config->set('chunk', 50);
      }

      // Set bundle dependency.
      $bundleEntityType = $this->entityTypeManager
        ->getDefinition($entityType)
        ->getBundleEntityType();
      if ($bundleEntityType) {
        $config->set('dependencies.config', ["{$bundleEntityType}.{$bundle}"]);
      }

      $config->save();

      $result = new ExecutionResult(TRUE);
      $result->addMessage($this->t('Auto entity label @action for @entity_type.@bundle.', [
        '@action' => $wasNew ? 'configured' : 'updated',
        '@entity_type' => $entityType,
        '@bundle' => $bundle,
      ]));

      $result->addAffectedEntity([
        'type' => 'config',
        'id' => $configName,
        'label' => "Auto label for {$entityType}.{$bundle}",
      ]);

      $result->setRollbackData([
        'config_name' => $configName,
        'was_new' => $wasNew,
        'original_data' => $originalData,
      ]);

      $this->logInfo('Configured auto entity label: @config', [
        '@config' => $configName,
      ]);

      return $result;
    }
    catch (\Exception $e) {
      throw new ExecutionException($e->getMessage(), [], 0, $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function rollback(): RollbackResult {
    try {
      $rollbackData = $this->getDataValue('_rollback_data', []);
      // Get config_name from rollback data first, fall back to building it.
      $configName = $rollbackData['config_name'] ?? NULL;
      if (empty($configName)) {
        $entityType = $this->getDataValue('entity_type');
        $bundle = $this->getDataValue('bundle');
        if (empty($entityType) || empty($bundle)) {
          $result = new RollbackResult(FALSE);
          $result->addMessage($this->t('Cannot rollback auto entity label: config_name is missing from rollback data.'));
          return $result;
        }
        $configName = "auto_entitylabel.settings.{$entityType}.{$bundle}";
      }
      $wasNew = $rollbackData['was_new'] ?? FALSE;
      $originalConfigData = $rollbackData['original_data'] ?? NULL;

      $config = $this->configFactory->getEditable($configName);

      if ($wasNew) {
        // Delete the newly created config.
        $config->delete();

        $result = new RollbackResult(TRUE);
        $result->addMessage($this->t('Auto entity label config "@name" deleted.', [
          '@name' => $configName,
        ]));
      }
      elseif ($originalConfigData !== NULL) {
        // Restore original values.
        foreach ($originalConfigData as $key => $value) {
          $config->set($key, $value);
        }
        $config->save();

        $result = new RollbackResult(TRUE);
        $result->addMessage($this->t('Auto entity label config "@name" restored.', [
          '@name' => $configName,
        ]));
      }
      else {
        $result = new RollbackResult(FALSE);
        $result->addMessage($this->t('No original data available for rollback of "@name".', [
          '@name' => $configName,
        ]));
      }

      $result->addRestoredEntity([
        'type' => 'config',
        'id' => $configName,
      ]);

      $this->logInfo('Rolled back auto entity label config: @config', [
        '@config' => $configName,
      ]);

      return $result;
    }
    catch (\Exception $e) {
      throw new RollbackException($e->getMessage(), [], 0, $e);
    }
  }

}
