<?php

namespace Drupal\Tests\eb_auto_entitylabel\Unit\Plugin\EbExtension;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\eb_auto_entitylabel\Plugin\EbExtension\AutoEntityLabelExtension;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for the AutoEntityLabelExtension plugin.
 *
 * @coversDefaultClass \Drupal\eb_auto_entitylabel\Plugin\EbExtension\AutoEntityLabelExtension
 * @group eb_auto_entitylabel
 */
class AutoEntityLabelExtensionTest extends UnitTestCase {

  /**
   * The extension plugin under test.
   */
  protected AutoEntityLabelExtension $extension;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->extension = new AutoEntityLabelExtension(
      [],
      'auto_entitylabel',
      [
        'id' => 'auto_entitylabel',
        'label' => 'Auto Entity Label',
        'description' => 'Provides automatic label pattern configuration via YAML',
        'yaml_keys' => ['auto_entitylabel_status', 'auto_entitylabel_pattern'],
        'operations' => ['configure_auto_entitylabel'],
        'module_dependencies' => ['auto_entitylabel'],
      ]
    );

    // Mock the string translation service.
    $translationMock = $this->createMock(TranslationInterface::class);
    $translationMock->method('translateString')
      ->willReturnCallback(function (TranslatableMarkup $string) {
        return $string->getUntranslatedString();
      });
    $this->extension->setStringTranslation($translationMock);
  }

  /**
   * Tests buildOperations with valid auto_entitylabel configuration.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsWithAutoEntityLabel(): void {
    $data = [
      'bundle_definitions' => [
        [
          'entity_type' => 'node',
          'bundle_id' => 'event',
          'label' => 'Event',
          'auto_entitylabel_status' => 'enabled',
          'auto_entitylabel_pattern' => '[node:field_date] - [node:field_name]',
          'auto_entitylabel_escape' => TRUE,
          'auto_entitylabel_preserve_titles' => FALSE,
          'auto_entitylabel_behavior' => 'after_save',
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertCount(1, $operations);
    $this->assertEquals('configure_auto_entitylabel', $operations[0]['operation']);
    $this->assertEquals('node', $operations[0]['entity_type']);
    $this->assertEquals('event', $operations[0]['bundle']);
    $this->assertEquals(1, $operations[0]['status']);
    $this->assertEquals('[node:field_date] - [node:field_name]', $operations[0]['pattern']);
    $this->assertTrue($operations[0]['escape']);
    $this->assertFalse($operations[0]['preserve_titles']);
    $this->assertEquals(1, $operations[0]['new_content_behavior']);
  }

  /**
   * Tests buildOperations maps status strings correctly.
   *
   * @covers ::buildOperations
   * @dataProvider statusMappingProvider
   */
  public function testBuildOperationsStatusMapping(string $statusString, int $expectedValue): void {
    $data = [
      'bundle_definitions' => [
        [
          'entity_type' => 'node',
          'bundle_id' => 'test',
          'auto_entitylabel_status' => $statusString,
          'auto_entitylabel_pattern' => '[node:title]',
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertEquals($expectedValue, $operations[0]['status']);
  }

  /**
   * Data provider for status mapping tests.
   *
   * @return array<string, array{string, int}>
   */
  public static function statusMappingProvider(): array {
    return [
      'disabled' => ['disabled', 0],
      'enabled' => ['enabled', 1],
      'optional' => ['optional', 2],
      'prefilled' => ['prefilled', 3],
    ];
  }

  /**
   * Tests buildOperations maps behavior strings correctly.
   *
   * @covers ::buildOperations
   * @dataProvider behaviorMappingProvider
   */
  public function testBuildOperationsBehaviorMapping(string $behaviorString, int $expectedValue): void {
    $data = [
      'bundle_definitions' => [
        [
          'entity_type' => 'node',
          'bundle_id' => 'test',
          'auto_entitylabel_status' => 'enabled',
          'auto_entitylabel_pattern' => '[node:title]',
          'auto_entitylabel_behavior' => $behaviorString,
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertEquals($expectedValue, $operations[0]['new_content_behavior']);
  }

  /**
   * Data provider for behavior mapping tests.
   *
   * @return array<string, array{string, int}>
   */
  public static function behaviorMappingProvider(): array {
    return [
      'before_save' => ['before_save', 0],
      'after_save' => ['after_save', 1],
    ];
  }

  /**
   * Tests buildOperations uses defaults when not provided.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsDefaults(): void {
    $data = [
      'bundle_definitions' => [
        [
          'entity_type' => 'node',
          'bundle_id' => 'test',
          'auto_entitylabel_status' => 'enabled',
          'auto_entitylabel_pattern' => '[node:title]',
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertFalse($operations[0]['escape']);
    $this->assertFalse($operations[0]['preserve_titles']);
    $this->assertEquals(0, $operations[0]['new_content_behavior']);
  }

  /**
   * Tests buildOperations returns empty when no auto_entitylabel_status.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsNoAutoEntityLabel(): void {
    $data = [
      'bundle_definitions' => [
        [
          'entity_type' => 'node',
          'bundle_id' => 'article',
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertEmpty($operations);
  }

  /**
   * Tests buildOperations returns empty when no bundle_definitions.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsEmpty(): void {
    $this->assertEmpty($this->extension->buildOperations([]));
    $this->assertEmpty($this->extension->buildOperations(['bundle_definitions' => []]));
  }

  /**
   * Tests buildOperations skips items without bundle_id.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsSkipsMissingBundleId(): void {
    $data = [
      'bundle_definitions' => [
        [
          'entity_type' => 'node',
          'auto_entitylabel_status' => 'enabled',
          'auto_entitylabel_pattern' => '[node:title]',
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertEmpty($operations);
  }

  /**
   * Tests buildOperations with multiple bundles.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsMultiple(): void {
    $data = [
      'bundle_definitions' => [
        [
          'entity_type' => 'node',
          'bundle_id' => 'article',
          'auto_entitylabel_status' => 'enabled',
          'auto_entitylabel_pattern' => '[node:title]',
        ],
        [
          'entity_type' => 'node',
          'bundle_id' => 'page',
          'auto_entitylabel_status' => 'disabled',
        ],
        [
          'entity_type' => 'node',
          'bundle_id' => 'event',
          // No auto_entitylabel_status - should be skipped.
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertCount(2, $operations);
    $this->assertEquals('article', $operations[0]['bundle']);
    $this->assertEquals('page', $operations[1]['bundle']);
  }

  /**
   * Tests getOperationDependencies returns bundle dependency.
   *
   * @covers ::getOperationDependencies
   */
  public function testGetOperationDependencies(): void {
    $operation = [
      'operation' => 'configure_auto_entitylabel',
      'entity_type' => 'node',
      'bundle' => 'event',
      'status' => 1,
    ];

    $dependencies = $this->extension->getOperationDependencies($operation, []);

    $this->assertContains('bundle:node:event', $dependencies);
  }

  /**
   * Tests getOperationDependencies returns empty for missing data.
   *
   * @covers ::getOperationDependencies
   */
  public function testGetOperationDependenciesEmpty(): void {
    $operation = ['operation' => 'configure_auto_entitylabel'];

    $dependencies = $this->extension->getOperationDependencies($operation, []);

    $this->assertEmpty($dependencies);
  }

  /**
   * Tests appliesTo returns true for auto_entitylabel operations.
   *
   * @covers ::appliesTo
   */
  public function testAppliesToAutoEntityLabelOperations(): void {
    $this->assertTrue($this->extension->appliesTo(['operation' => 'configure_auto_entitylabel']));
  }

  /**
   * Tests appliesTo returns false for other operations.
   *
   * @covers ::appliesTo
   */
  public function testAppliesToOtherOperations(): void {
    $this->assertFalse($this->extension->appliesTo(['operation' => 'create_bundle']));
    $this->assertFalse($this->extension->appliesTo(['operation' => 'create_field']));
  }

  /**
   * Tests getYamlKeys returns correct keys.
   *
   * @covers ::getYamlKeys
   */
  public function testGetYamlKeys(): void {
    $expected = ['auto_entitylabel_status', 'auto_entitylabel_pattern'];
    $this->assertEquals($expected, $this->extension->getYamlKeys());
  }

  /**
   * Tests getOperations returns correct operations.
   *
   * @covers ::getOperations
   */
  public function testGetOperations(): void {
    $this->assertEquals(['configure_auto_entitylabel'], $this->extension->getOperations());
  }

}
