<?php

namespace Drupal\eb_field_group\Plugin\EbExtension;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eb\Attribute\EbExtension;
use Drupal\eb\PluginBase\EbExtensionBase;

/**
 * Extension plugin for field group support in Entity Builder.
 *
 * Integrates the field_group contrib module with Entity Builder, allowing
 * field groups to be defined in YAML and managed through Entity Builder's
 * operation system.
 *
 * Example YAML:
 *
 * field_group_definitions:
 *   - entity_type: node
 *     bundle: article
 *     display_type: form
 *     mode: default
 *     group_name: group_content
 *     label: Content
 *     format_type: fieldset
 *     weight: 0
 *     parent: ""
 *     children: [field_body, field_summary]
 *
 * display_field_definitions:
 *   - entity_type: node
 *     bundle: article
 *     display_type: form
 *     mode: default
 *     field_name: field_body
 *     group: group_content
 */
#[EbExtension(
  id: 'field_group',
  label: new TranslatableMarkup('Field Group'),
  description: new TranslatableMarkup('Organize fields into groups on entity forms and displays using the field_group module.'),
  yaml_keys: ['field_group_definitions'],
  operations: ['create_field_group', 'update_field_group', 'delete_field_group'],
  module_dependencies: ['field_group'],
)]
class FieldGroupExtension extends EbExtensionBase {

  /**
   * {@inheritdoc}
   */
  public function buildOperations(array $data): array {
    $fieldGroupDefinitions = $data['field_group_definitions'] ?? [];
    if (!is_array($fieldGroupDefinitions)) {
      $fieldGroupDefinitions = [];
    }

    $operations = [];
    $groupChildren = [];

    // 1. Build operations from field_group_definitions.
    foreach ($fieldGroupDefinitions as $group) {
      if (!is_array($group) || empty($group['group_name'])) {
        continue;
      }

      $entityType = $group['entity_type'] ?? '';
      $bundle = $group['bundle'] ?? '';
      $displayType = $group['display_type'] ?? '';
      $mode = $group['mode'] ?? 'default';

      if (empty($entityType) || empty($bundle) || empty($displayType)) {
        continue;
      }

      // Create a unique key for this group.
      $groupKey = "$entityType:$bundle:$displayType:$mode:{$group['group_name']}";

      $operations[] = [
        'operation' => 'create_field_group',
        'entity_type' => $entityType,
        'bundle' => $bundle,
        'display_type' => $displayType,
        'display_id' => $mode,
        'group_name' => $group['group_name'],
        'label' => $group['label'] ?? $group['group_name'],
        'format_type' => $group['format_type'] ?? 'fieldset',
        'format_settings' => $group['format_settings'] ?? [],
        'children' => $group['children'] ?? [],
        'parent' => $group['parent'] ?? '',
        'weight' => $group['weight'] ?? 0,
        '_group_key' => $groupKey,
      ];

      // Initialize children tracking from existing children.
      $groupChildren[$groupKey] = $group['children'] ?? [];
    }

    // 2. Collect form_group/view_group from field_definitions.
    $fieldDefinitions = $data['field_definitions'] ?? [];
    if (is_array($fieldDefinitions)) {
      foreach ($fieldDefinitions as $field) {
        if (!is_array($field)) {
          continue;
        }

        $fieldName = $field['field_name'] ?? '';
        $entityType = $field['entity_type'] ?? '';
        $bundle = $field['bundle'] ?? '';

        if (empty($fieldName) || empty($entityType) || empty($bundle)) {
          continue;
        }

        // Handle form_group assignment.
        if (!empty($field['form_group'])) {
          $formGroupKey = "$entityType:$bundle:form:default:{$field['form_group']}";
          if (!isset($groupChildren[$formGroupKey])) {
            $groupChildren[$formGroupKey] = [];
          }
          if (!in_array($fieldName, $groupChildren[$formGroupKey], TRUE)) {
            $groupChildren[$formGroupKey][] = $fieldName;
          }
        }

        // Handle view_group assignment.
        if (!empty($field['view_group'])) {
          $viewGroupKey = "$entityType:$bundle:view:default:{$field['view_group']}";
          if (!isset($groupChildren[$viewGroupKey])) {
            $groupChildren[$viewGroupKey] = [];
          }
          if (!in_array($fieldName, $groupChildren[$viewGroupKey], TRUE)) {
            $groupChildren[$viewGroupKey][] = $fieldName;
          }
        }
      }
    }

    // 3. Update operations with collected children from field_definitions.
    foreach ($operations as &$op) {
      $groupKey = $op['_group_key'];
      if (isset($groupChildren[$groupKey])) {
        $op['children'] = array_values(array_unique($groupChildren[$groupKey]));
      }
      // Remove the internal key.
      unset($op['_group_key']);
    }

    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperationDependencies(array $operation, array $context): array {
    $dependencies = [];
    $entityType = $operation['entity_type'] ?? '';
    $bundle = $operation['bundle'] ?? '';
    $displayType = $operation['display_type'] ?? '';
    $displayId = $operation['display_id'] ?? '';

    // Depend on bundle.
    if ($entityType && $bundle) {
      $dependencies[] = "bundle:$entityType:$bundle";
    }

    // Depend on display configuration.
    if ($entityType && $bundle && $displayType && $displayId) {
      $dependencies[] = "display:$entityType:$bundle:$displayType:$displayId";
    }

    // Depend on child fields.
    foreach ($operation['children'] ?? [] as $fieldName) {
      if ($fieldName && $entityType && $bundle) {
        $dependencies[] = "field:$entityType:$bundle:$fieldName";
      }
    }

    // Depend on parent group if nested.
    if (!empty($operation['parent']) && $entityType && $bundle && $displayType && $displayId) {
      $dependencies[] = "field_group:$entityType:$bundle:$displayType:$displayId:{$operation['parent']}";
    }

    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public function detectChanges(array $operation, array $context): ?string {
    if (($operation['operation'] ?? '') !== 'create_field_group') {
      return NULL;
    }

    $entityType = $operation['entity_type'] ?? '';
    $bundle = $operation['bundle'] ?? '';
    $displayType = $operation['display_type'] ?? '';
    $displayId = $operation['display_id'] ?? '';
    $groupName = $operation['group_name'] ?? '';

    if (!$entityType || !$bundle || !$displayType || !$groupName) {
      return NULL;
    }

    if (!function_exists('field_group_load_field_group')) {
      return NULL;
    }

    // Mode mapping: 'view' => 'display', 'form' => 'form'.
    $mode = $displayType === 'view' ? 'display' : 'form';

    $existing = field_group_load_field_group($groupName, $entityType, $bundle, $mode, $displayId);

    if (empty($existing)) {
      return 'create_field_group';
    }

    // Check for changes.
    if (($existing->label ?? '') !== ($operation['label'] ?? '')) {
      return 'update_field_group';
    }
    if (($existing->format_type ?? '') !== ($operation['format_type'] ?? 'fieldset')) {
      return 'update_field_group';
    }
    if (($existing->weight ?? 0) !== ($operation['weight'] ?? 0)) {
      return 'update_field_group';
    }
    if (($existing->parent_name ?? '') !== ($operation['parent'] ?? '')) {
      return 'update_field_group';
    }

    // Compare children.
    $existingChildren = $existing->children ?? [];
    $newChildren = $operation['children'] ?? [];
    sort($existingChildren);
    sort($newChildren);
    if ($existingChildren !== $newChildren) {
      return 'update_field_group';
    }

    // Compare format_settings.
    if (isset($operation['format_settings'])) {
      $existingSettings = $existing->format_settings ?? [];
      if (serialize($existingSettings) !== serialize($operation['format_settings'])) {
        return 'update_field_group';
      }
    }

    return 'skip';
  }

  /**
   * {@inheritdoc}
   */
  public function checkDependencies(array $dependencyKeys): array {
    $results = [];

    foreach ($dependencyKeys as $key) {
      $parts = explode(':', $key);

      // Only handle field_group dependencies.
      if ($parts[0] !== 'field_group') {
        continue;
      }

      // Parse: field_group:{entity_type}:{bundle}:{display_type}:{display_id}:{group_name}.
      if (count($parts) < 6) {
        continue;
      }

      [, $entityType, $bundle, $displayType, $displayId, $groupName] = $parts;

      if (!$entityType || !$bundle || !$displayType || !$displayId || !$groupName) {
        continue;
      }

      if (!function_exists('field_group_load_field_group')) {
        continue;
      }

      $mode = $displayType === 'view' ? 'display' : 'form';
      $group = field_group_load_field_group($groupName, $entityType, $bundle, $mode, $displayId);
      $results[$key] = !empty($group);
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function extractConfig(string $entityTypeId, string $bundle): array {
    if (!function_exists('field_group_info_groups')) {
      return [];
    }

    $groups = field_group_info_groups($entityTypeId, $bundle, 'form', 'default');
    if (empty($groups)) {
      return [];
    }

    $definitions = [];
    foreach ($groups as $group) {
      $definitions[] = [
        'entity_type' => $entityTypeId,
        'bundle' => $bundle,
        'display_type' => 'form',
        'mode' => 'default',
        'group_name' => $group->group_name,
        'label' => $group->label,
        'format_type' => $group->format_type,
        'weight' => $group->weight,
        'children' => $group->children,
        'format_settings' => $group->format_settings ?? [],
      ];
    }

    return ['field_group_definitions' => $definitions];
  }

}
