<?php

namespace Drupal\eb_field_group\Plugin\EbOperation;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eb\Attribute\EbOperation;
use Drupal\eb\Exception\ExecutionException;
use Drupal\eb\PluginBase\OperationBase;
use Drupal\eb\Result\ExecutionResult;
use Drupal\eb\Result\PreviewResult;
use Drupal\eb\Result\RollbackResult;
use Drupal\eb\Result\ValidationResult;
use Drupal\field_group\FieldGroupFormatterPluginManager;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Operation for creating field groups on form or view displays.
 */
#[EbOperation(
    id: 'create_field_group',
    label: new TranslatableMarkup('Create Field Group'),
    description: new TranslatableMarkup('Creates a field group on a form or view display'),
    operationType: 'create',
)]
class CreateFieldGroupOperation extends OperationBase {

  /**
   * Valid field group format types.
   */
  protected const VALID_FORMAT_TYPES = [
    'fieldset',
    'details',
    'tabs',
    'tab',
    'accordion',
    'accordion_item',
    'html_element',
  ];

  /**
   * Stores the original field group for rollback.
   *
   * @var object|null
   */
  protected ?object $originalGroup = NULL;

  /**
   * Constructor.
   *
   * @param array<string, mixed> $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger channel.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   Module handler.
   * @param \Drupal\field_group\FieldGroupFormatterPluginManager|null $formatterManager
   *   Field group formatter plugin manager.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
    LoggerInterface $logger,
    ConfigFactoryInterface $configFactory,
    protected ModuleHandlerInterface $moduleHandler,
    protected ?FieldGroupFormatterPluginManager $formatterManager = NULL,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityTypeManager, $logger, $configFactory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    // Check if field_group module is installed (service may not exist).
    // @phpstan-ignore ternary.alwaysTrue
    $formatterManager = $container->has('plugin.manager.field_group.formatters')
      ? $container->get('plugin.manager.field_group.formatters')
      : NULL;

    /** @var \Psr\Log\LoggerInterface $logger */
    $logger = $container->get('logger.channel.eb');

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $logger,
      $container->get('config.factory'),
      $container->get('module_handler'),
      $formatterManager,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate(): ValidationResult {
    $result = new ValidationResult();

    // Validate required fields.
    $this->validateRequiredFields(
          ['group_name', 'entity_type', 'bundle', 'display_type', 'display_id', 'label'],
          $result
      );

    if (!$result->isValid()) {
      return $result;
    }

    // Validate format_type if provided.
    $formatType = $this->getDataValue('format_type', 'fieldset');
    if (!in_array($formatType, self::VALID_FORMAT_TYPES, TRUE)) {
      $result->addError(
            'format_type',
            $this->t(
                'Invalid format_type "@type". Must be one of: @types',
                [
                  '@type' => $formatType,
                  '@types' => implode(', ', self::VALID_FORMAT_TYPES),
                ]
            )
        );
    }

    // Validate children is array if present.
    $children = $this->getDataValue('children', []);
    if (!is_array($children)) {
      $result->addError(
            'children',
            $this->t('Children must be an array')
        );
    }

    // Validate format_settings is array if present.
    $formatSettings = $this->getDataValue('format_settings', []);
    if (!is_array($formatSettings)) {
      $result->addError(
            'format_settings',
            $this->t('Format settings must be an array')
        );
    }

    // Validate display_type is 'form' or 'view'.
    $displayType = $this->getDataValue('display_type');
    if ($displayType !== 'form' && $displayType !== 'view') {
      $result->addError(
            'display_type',
            $this->t('Display type must be "form" or "view"')
        );
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function preview(): PreviewResult {
    $group_name = $this->getDataValue('group_name');
    $entity_type = $this->getDataValue('entity_type');
    $bundle = $this->getDataValue('bundle');
    $display_type = $this->getDataValue('display_type');
    $display_id = $this->getDataValue('display_id');
    $label = $this->getDataValue('label');
    $format_type = $this->getDataValue('format_type', 'fieldset');

    return new PreviewResult(
      $this->t(
        'Create field group "@name" (@label) on @entity_type:@bundle @display_type display (@mode) with format @format',
        [
          '@name' => $group_name,
          '@label' => $label,
          '@entity_type' => $entity_type,
          '@bundle' => $bundle,
          '@display_type' => $display_type,
          '@mode' => $display_id,
          '@format' => $format_type,
        ]
      )->__toString()
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): ExecutionResult {
    // Check if field_group module is installed.
    if (!$this->moduleHandler->moduleExists('field_group')) {
      throw new ExecutionException(
            'Field group module is not installed. Please run: composer require drupal/field_group'
        );
    }

    $entityType = $this->getDataValue('entity_type');
    $bundle = $this->getDataValue('bundle');
    $displayType = $this->getDataValue('display_type');
    $displayId = $this->getDataValue('display_id');
    $groupName = $this->getDataValue('group_name');
    $label = $this->getDataValue('label');
    $formatType = $this->getDataValue('format_type', 'fieldset');
    $formatSettings = $this->getDataValue('format_settings', []);
    $children = $this->getDataValue('children', []);
    $parentName = $this->getDataValue('parent', '');
    $weight = $this->getDataValue('weight', 0);

    // Get default format settings from the formatter plugin.
    if ($this->formatterManager) {
      $defaultSettings = $this->formatterManager->getDefaultSettings($formatType, $displayType);
      $formatSettings = array_merge($defaultSettings, $formatSettings);
    }

    // Build field group structure.
    $group = (object) [
      'group_name' => $groupName,
      'entity_type' => $entityType,
      'bundle' => $bundle,
      'mode' => $displayId,
      'context' => $displayType,
      'children' => $children,
      'parent_name' => $parentName,
      'weight' => $weight,
      'label' => $label,
      'format_type' => $formatType,
      'format_settings' => $formatSettings,
      'region' => 'content',
    ];

    // Save the field group.
    field_group_group_save($group);

    // Store for rollback.
    $this->originalGroup = $group;

    $this->logInfo(
          'Created field group @name on @entity_type:@bundle @display_type display',
          [
            '@name' => $groupName,
            '@entity_type' => $entityType,
            '@bundle' => $bundle,
            '@display_type' => $displayType,
          ]
      );

    return new ExecutionResult(
          TRUE,
          [],
          [],
          [$this->t(
              'Successfully created field group "@name"',
              ['@name' => $groupName]
            )->__toString(),
          ]
      );
  }

  /**
   * {@inheritdoc}
   */
  public function rollback(): RollbackResult {
    if (!$this->moduleHandler->moduleExists('field_group')) {
      return new RollbackResult(
            FALSE,
            [],
            [$this->t('Field group module not installed, cannot rollback')->__toString()]
        );
    }

    $entityType = $this->getDataValue('entity_type');
    $bundle = $this->getDataValue('bundle');
    $displayType = $this->getDataValue('display_type');
    $displayId = $this->getDataValue('display_id');
    $groupName = $this->getDataValue('group_name');

    // Load the field group.
    $group = field_group_load_field_group(
          $groupName,
          $entityType,
          $bundle,
          $displayType,
          $displayId
      );

    if ($group) {
      // @phpstan-ignore function.notFound
      field_group_group_export_delete($group, FALSE);

      $this->logInfo(
            'Rolled back creation of field group @name',
            ['@name' => $groupName]
        );

      return new RollbackResult(
            TRUE,
            [],
            [$this->t(
                'Successfully removed field group "@name"',
                ['@name' => $groupName]
              )->__toString(),
            ]
        );
    }

    return new RollbackResult(
          FALSE,
          [],
          [$this->t(
              'Field group "@name" not found for rollback',
              ['@name' => $groupName]
            )->__toString(),
          ]
      );
  }

}
