<?php

namespace Drupal\eb_field_group\Plugin\EbOperation;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eb\Attribute\EbOperation;
use Drupal\eb\Exception\ExecutionException;
use Drupal\eb\PluginBase\OperationBase;
use Drupal\eb\Result\ExecutionResult;
use Drupal\eb\Result\PreviewResult;
use Drupal\eb\Result\RollbackResult;
use Drupal\eb\Result\ValidationResult;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Operation for deleting field groups.
 */
#[EbOperation(
    id: 'delete_field_group',
    label: new TranslatableMarkup('Delete Field Group'),
    description: new TranslatableMarkup('Deletes a field group from a form or view display'),
    operationType: 'delete',
)]
class DeleteFieldGroupOperation extends OperationBase {

  /**
   * Stores the original field group for rollback.
   *
   * @var object|null
   */
  protected ?object $originalGroup = NULL;

  /**
   * Constructor.
   *
   * @param array<string, mixed> $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger channel.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   Module handler.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
    LoggerInterface $logger,
    ConfigFactoryInterface $configFactory,
    protected ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityTypeManager, $logger, $configFactory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Psr\Log\LoggerInterface $logger */
    $logger = $container->get('logger.channel.eb');

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $logger,
      $container->get('config.factory'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate(): ValidationResult {
    $result = new ValidationResult();

    // Validate required fields.
    $this->validateRequiredFields(
          ['group_name', 'entity_type', 'bundle', 'display_type', 'display_id'],
          $result
      );

    if (!$result->isValid()) {
      return $result;
    }

    // Validate display_type is 'form' or 'view'.
    $displayType = $this->getDataValue('display_type');
    if ($displayType !== 'form' && $displayType !== 'view') {
      $result->addError(
            'display_type',
            $this->t('Display type must be "form" or "view"')
        );
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function preview(): PreviewResult {
    $group_name = $this->getDataValue('group_name');
    $entity_type = $this->getDataValue('entity_type');
    $bundle = $this->getDataValue('bundle');
    $display_type = $this->getDataValue('display_type');
    $display_id = $this->getDataValue('display_id');

    return new PreviewResult(
      $this->t(
        'Delete field group "@name" from @entity_type:@bundle @display_type display (@mode)',
        [
          '@name' => $group_name,
          '@entity_type' => $entity_type,
          '@bundle' => $bundle,
          '@display_type' => $display_type,
          '@mode' => $display_id,
        ]
      )->__toString()
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): ExecutionResult {
    // Check if field_group module is installed.
    if (!$this->moduleHandler->moduleExists('field_group')) {
      throw new ExecutionException(
            'Field group module is not installed. Please run: composer require drupal/field_group'
        );
    }

    $entityType = $this->getDataValue('entity_type');
    $bundle = $this->getDataValue('bundle');
    $displayType = $this->getDataValue('display_type');
    $displayId = $this->getDataValue('display_id');
    $groupName = $this->getDataValue('group_name');

    // Load existing field group.
    $existingGroup = field_group_load_field_group(
          $groupName,
          $entityType,
          $bundle,
          $displayType,
          $displayId
      );

    if (!$existingGroup) {
      throw new ExecutionException(
            sprintf(
                'Field group "%s" does not exist on %s:%s %s display',
                $groupName,
                $entityType,
                $bundle,
                $displayType
            )
        );
    }

    // Store original for rollback.
    $this->originalGroup = clone $existingGroup;

    // Delete the field group.
    // @phpstan-ignore function.notFound
    field_group_group_export_delete($existingGroup, FALSE);

    $this->logInfo(
          'Deleted field group @name from @entity_type:@bundle @display_type display',
          [
            '@name' => $groupName,
            '@entity_type' => $entityType,
            '@bundle' => $bundle,
            '@display_type' => $displayType,
          ]
      );

    return new ExecutionResult(
          TRUE,
          [],
          [],
          [$this->t(
              'Successfully deleted field group "@name"',
              ['@name' => $groupName]
            )->__toString(),
          ]
      );
  }

  /**
   * {@inheritdoc}
   */
  public function rollback(): RollbackResult {
    if (!$this->moduleHandler->moduleExists('field_group')) {
      return new RollbackResult(
            FALSE,
            [],
            [$this->t('Field group module not installed, cannot rollback')->__toString()]
        );
    }

    if (!$this->originalGroup) {
      return new RollbackResult(
            FALSE,
            [],
            [$this->t('No original field group stored for rollback')->__toString()]
        );
    }

    // Recreate the deleted field group.
    field_group_group_save($this->originalGroup);

    $groupName = $this->originalGroup->group_name;

    $this->logInfo(
          'Rolled back deletion of field group @name',
          ['@name' => $groupName]
      );

    return new RollbackResult(
          TRUE,
          [],
          [$this->t(
              'Successfully restored field group "@name"',
              ['@name' => $groupName]
            )->__toString(),
          ]
      );
  }

}
