<?php

namespace Drupal\eb_field_group\Plugin\EbOperation;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eb\Attribute\EbOperation;
use Drupal\eb\Exception\ExecutionException;
use Drupal\eb\PluginBase\OperationBase;
use Drupal\eb\Result\ExecutionResult;
use Drupal\eb\Result\PreviewResult;
use Drupal\eb\Result\RollbackResult;
use Drupal\eb\Result\ValidationResult;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Operation for updating existing field groups.
 */
#[EbOperation(
    id: 'update_field_group',
    label: new TranslatableMarkup('Update Field Group'),
    description: new TranslatableMarkup('Updates an existing field group on a form or view display'),
    operationType: 'update',
)]
class UpdateFieldGroupOperation extends OperationBase {

  /**
   * Valid field group format types.
   */
  protected const VALID_FORMAT_TYPES = [
    'fieldset',
    'details',
    'tabs',
    'tab',
    'accordion',
    'accordion_item',
    'html_element',
  ];

  /**
   * Stores the original field group for rollback.
   *
   * @var object|null
   */
  protected ?object $originalGroup = NULL;

  /**
   * Constructor.
   *
   * @param array<string, mixed> $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger channel.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   Module handler.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
    LoggerInterface $logger,
    ConfigFactoryInterface $configFactory,
    protected ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entityTypeManager, $logger, $configFactory);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Psr\Log\LoggerInterface $logger */
    $logger = $container->get('logger.channel.eb');

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $logger,
      $container->get('config.factory'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate(): ValidationResult {
    $result = new ValidationResult();

    // Validate required fields.
    $this->validateRequiredFields(
          ['group_name', 'entity_type', 'bundle', 'display_type', 'display_id', 'label'],
          $result
      );

    if (!$result->isValid()) {
      return $result;
    }

    // Validate format_type if provided.
    $formatType = $this->getDataValue('format_type', 'fieldset');
    if (!in_array($formatType, self::VALID_FORMAT_TYPES, TRUE)) {
      $result->addError(
            'format_type',
            $this->t(
                'Invalid format_type "@type". Must be one of: @types',
                [
                  '@type' => $formatType,
                  '@types' => implode(', ', self::VALID_FORMAT_TYPES),
                ]
            )
        );
    }

    // Validate children is array if present.
    $children = $this->getDataValue('children', []);
    if (!is_array($children)) {
      $result->addError(
            'children',
            $this->t('Children must be an array')
        );
    }

    // Validate format_settings is array if present.
    $formatSettings = $this->getDataValue('format_settings', []);
    if (!is_array($formatSettings)) {
      $result->addError(
            'format_settings',
            $this->t('Format settings must be an array')
        );
    }

    // Validate display_type is 'form' or 'view'.
    $displayType = $this->getDataValue('display_type');
    if ($displayType !== 'form' && $displayType !== 'view') {
      $result->addError(
            'display_type',
            $this->t('Display type must be "form" or "view"')
        );
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function preview(): PreviewResult {
    $group_name = $this->getDataValue('group_name');
    $entity_type = $this->getDataValue('entity_type');
    $bundle = $this->getDataValue('bundle');
    $display_type = $this->getDataValue('display_type');
    $display_id = $this->getDataValue('display_id');
    $label = $this->getDataValue('label');

    return new PreviewResult(
      $this->t(
        'Update field group "@name" (@label) on @entity_type:@bundle @display_type display (@mode)',
        [
          '@name' => $group_name,
          '@label' => $label,
          '@entity_type' => $entity_type,
          '@bundle' => $bundle,
          '@display_type' => $display_type,
          '@mode' => $display_id,
        ]
      )->__toString()
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): ExecutionResult {
    // Check if field_group module is installed.
    if (!$this->moduleHandler->moduleExists('field_group')) {
      throw new ExecutionException(
            'Field group module is not installed. Please run: composer require drupal/field_group'
        );
    }

    $entityType = $this->getDataValue('entity_type');
    $bundle = $this->getDataValue('bundle');
    $displayType = $this->getDataValue('display_type');
    $displayId = $this->getDataValue('display_id');
    $groupName = $this->getDataValue('group_name');

    // Load existing field group.
    $existingGroup = field_group_load_field_group(
          $groupName,
          $entityType,
          $bundle,
          $displayType,
          $displayId
      );

    if (!$existingGroup) {
      throw new ExecutionException(
            sprintf(
                'Field group "%s" does not exist on %s:%s %s display',
                $groupName,
                $entityType,
                $bundle,
                $displayType
            )
        );
    }

    // Store original for rollback.
    $this->originalGroup = clone $existingGroup;

    // Update properties from operation data.
    $label = $this->getDataValue('label');
    $formatType = $this->getDataValue('format_type', 'fieldset');
    $formatSettings = $this->getDataValue('format_settings', []);
    $children = $this->getDataValue('children', []);
    $parentName = $this->getDataValue('parent', '');
    $weight = $this->getDataValue('weight', 0);

    // Get default format settings and merge.
    // @phpstan-ignore globalDrupalDependencyInjection.useDependencyInjection
    $formatterManager = \Drupal::service('plugin.manager.field_group.formatters');
    $defaultSettings = $formatterManager->getDefaultSettings($formatType, $displayType);
    $formatSettings = array_merge($defaultSettings, $formatSettings);

    // Update the group.
    $existingGroup->label = $label;
    $existingGroup->format_type = $formatType;
    $existingGroup->format_settings = $formatSettings;
    $existingGroup->children = $children;
    $existingGroup->parent_name = $parentName;
    $existingGroup->weight = $weight;

    // Save the updated group.
    field_group_group_save($existingGroup);

    $this->logInfo(
          'Updated field group @name on @entity_type:@bundle @display_type display',
          [
            '@name' => $groupName,
            '@entity_type' => $entityType,
            '@bundle' => $bundle,
            '@display_type' => $displayType,
          ]
      );

    return new ExecutionResult(
          TRUE,
          [],
          [],
          [$this->t(
              'Successfully updated field group "@name"',
              ['@name' => $groupName]
            )->__toString(),
          ]
      );
  }

  /**
   * {@inheritdoc}
   */
  public function rollback(): RollbackResult {
    if (!$this->moduleHandler->moduleExists('field_group')) {
      return new RollbackResult(
            FALSE,
            [],
            [$this->t('Field group module not installed, cannot rollback')->__toString()]
        );
    }

    if (!$this->originalGroup) {
      return new RollbackResult(
            FALSE,
            [],
            [$this->t('No original field group stored for rollback')->__toString()]
        );
    }

    // Restore the original field group.
    field_group_group_save($this->originalGroup);

    $groupName = $this->originalGroup->group_name;

    $this->logInfo(
          'Rolled back update of field group @name',
          ['@name' => $groupName]
      );

    return new RollbackResult(
          TRUE,
          [],
          [$this->t(
              'Successfully restored field group "@name"',
              ['@name' => $groupName]
            )->__toString(),
          ]
      );
  }

}
