<?php

namespace Drupal\Tests\eb_field_group\Unit\Plugin\EbExtension;

use Drupal\eb_field_group\Plugin\EbExtension\FieldGroupExtension;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for the FieldGroupExtension plugin.
 *
 * @coversDefaultClass \Drupal\eb_field_group\Plugin\EbExtension\FieldGroupExtension
 * @group eb_field_group
 */
class FieldGroupExtensionTest extends UnitTestCase {

  /**
   * The extension plugin under test.
   */
  protected FieldGroupExtension $extension;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->extension = new FieldGroupExtension(
      [],
      'field_group',
      [
        'id' => 'field_group',
        'label' => 'Field Group',
        'description' => 'Test description',
        'yaml_keys' => ['field_group_definitions'],
        'operations' => ['create_field_group', 'update_field_group', 'delete_field_group'],
        'module_dependencies' => ['field_group'],
      ]
    );
  }

  /**
   * Tests buildOperations with valid field group definitions.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperations(): void {
    $data = [
      'field_group_definitions' => [
        [
          'entity_type' => 'node',
          'bundle' => 'article',
          'display_type' => 'form',
          'mode' => 'default',
          'group_name' => 'group_main',
          'label' => 'Main Info',
          'format_type' => 'fieldset',
          'children' => ['field_title', 'field_body'],
          'weight' => 0,
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertCount(1, $operations);
    $this->assertEquals('create_field_group', $operations[0]['operation']);
    $this->assertEquals('node', $operations[0]['entity_type']);
    $this->assertEquals('article', $operations[0]['bundle']);
    $this->assertEquals('form', $operations[0]['display_type']);
    $this->assertEquals('default', $operations[0]['display_id']);
    $this->assertEquals('group_main', $operations[0]['group_name']);
    $this->assertEquals('Main Info', $operations[0]['label']);
    $this->assertEquals('fieldset', $operations[0]['format_type']);
    $this->assertEquals(['field_title', 'field_body'], $operations[0]['children']);
  }

  /**
   * Tests buildOperations with multiple groups.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsMultiple(): void {
    $data = [
      'field_group_definitions' => [
        [
          'entity_type' => 'node',
          'bundle' => 'page',
          'display_type' => 'form',
          'mode' => 'default',
          'group_name' => 'group_content',
          'label' => 'Content',
          'format_type' => 'details',
        ],
        [
          'entity_type' => 'node',
          'bundle' => 'page',
          'display_type' => 'view',
          'mode' => 'default',
          'group_name' => 'group_sidebar',
          'label' => 'Sidebar',
          'format_type' => 'fieldset',
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertCount(2, $operations);
    $this->assertEquals('form', $operations[0]['display_type']);
    $this->assertEquals('group_content', $operations[0]['group_name']);
    $this->assertEquals('view', $operations[1]['display_type']);
    $this->assertEquals('group_sidebar', $operations[1]['group_name']);
  }

  /**
   * Tests buildOperations returns empty when no definitions.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsEmpty(): void {
    $this->assertEmpty($this->extension->buildOperations([]));
    $this->assertEmpty($this->extension->buildOperations(['field_group_definitions' => []]));
  }

  /**
   * Tests buildOperations skips items without group_name.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsSkipsInvalid(): void {
    $data = [
      'field_group_definitions' => [
        [
          'entity_type' => 'node',
          'bundle' => 'article',
          'display_type' => 'form',
          'mode' => 'default',
          // Missing group_name.
          'label' => 'Test Label',
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertEmpty($operations);
  }

  /**
   * Tests buildOperations applies defaults.
   *
   * @covers ::buildOperations
   */
  public function testBuildOperationsAppliesDefaults(): void {
    $data = [
      'field_group_definitions' => [
        [
          'entity_type' => 'node',
          'bundle' => 'article',
          'display_type' => 'form',
          'group_name' => 'group_test',
        ],
      ],
    ];

    $operations = $this->extension->buildOperations($data);

    $this->assertCount(1, $operations);
    $this->assertEquals('default', $operations[0]['display_id']);
    $this->assertEquals('group_test', $operations[0]['label']);
    $this->assertEquals('fieldset', $operations[0]['format_type']);
    $this->assertEquals([], $operations[0]['children']);
    $this->assertEquals('', $operations[0]['parent']);
    $this->assertEquals(0, $operations[0]['weight']);
  }

  /**
   * Tests getOperationDependencies returns bundle dependency.
   *
   * @covers ::getOperationDependencies
   */
  public function testGetOperationDependenciesBundleDependency(): void {
    $operation = [
      'operation' => 'create_field_group',
      'entity_type' => 'node',
      'bundle' => 'article',
      'display_type' => 'form',
      'display_id' => 'default',
      'group_name' => 'group_test',
    ];

    $dependencies = $this->extension->getOperationDependencies($operation, []);

    $this->assertContains('bundle:node:article', $dependencies);
  }

  /**
   * Tests getOperationDependencies returns field dependencies.
   *
   * @covers ::getOperationDependencies
   */
  public function testGetOperationDependenciesFieldDependencies(): void {
    $operation = [
      'operation' => 'create_field_group',
      'entity_type' => 'node',
      'bundle' => 'article',
      'display_type' => 'form',
      'display_id' => 'default',
      'group_name' => 'group_test',
      'children' => ['field_title', 'field_body'],
    ];

    $dependencies = $this->extension->getOperationDependencies($operation, []);

    $this->assertContains('field:node:article:field_title', $dependencies);
    $this->assertContains('field:node:article:field_body', $dependencies);
  }

  /**
   * Tests getOperationDependencies returns parent group dependency.
   *
   * @covers ::getOperationDependencies
   */
  public function testGetOperationDependenciesParentDependency(): void {
    $operation = [
      'operation' => 'create_field_group',
      'entity_type' => 'node',
      'bundle' => 'article',
      'display_type' => 'form',
      'display_id' => 'default',
      'group_name' => 'group_child',
      'parent' => 'group_parent',
    ];

    $dependencies = $this->extension->getOperationDependencies($operation, []);

    $this->assertContains('field_group:node:article:form:default:group_parent', $dependencies);
  }

  /**
   * Tests getOperationDependencies returns display dependency.
   *
   * @covers ::getOperationDependencies
   */
  public function testGetOperationDependenciesDisplayDependency(): void {
    $operation = [
      'operation' => 'create_field_group',
      'entity_type' => 'node',
      'bundle' => 'article',
      'display_type' => 'form',
      'display_id' => 'default',
      'group_name' => 'group_test',
    ];

    $dependencies = $this->extension->getOperationDependencies($operation, []);

    $this->assertContains('display:node:article:form:default', $dependencies);
  }

  /**
   * Tests appliesTo returns true for field_group operations.
   *
   * @covers ::appliesTo
   */
  public function testAppliesToFieldGroupOperations(): void {
    $this->assertTrue($this->extension->appliesTo(['operation' => 'create_field_group']));
    $this->assertTrue($this->extension->appliesTo(['operation' => 'update_field_group']));
    $this->assertTrue($this->extension->appliesTo(['operation' => 'delete_field_group']));
  }

  /**
   * Tests appliesTo returns false for non-field_group operations.
   *
   * @covers ::appliesTo
   */
  public function testAppliesToOtherOperations(): void {
    $this->assertFalse($this->extension->appliesTo(['operation' => 'create_bundle']));
    $this->assertFalse($this->extension->appliesTo(['operation' => 'create_field']));
  }

  /**
   * Tests getYamlKeys returns expected keys.
   *
   * @covers ::getYamlKeys
   */
  public function testGetYamlKeys(): void {
    $this->assertEquals(['field_group_definitions'], $this->extension->getYamlKeys());
  }

  /**
   * Tests getOperations returns expected operations.
   *
   * @covers ::getOperations
   */
  public function testGetOperations(): void {
    $expected = ['create_field_group', 'update_field_group', 'delete_field_group'];
    $this->assertEquals($expected, $this->extension->getOperations());
  }

}
