<?php

namespace Drupal\eca_breadcrumbs\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;

/**
 * Set all breadcrumb items at once.
 *
 * @Action(
 *   id = "eca_breadcrumbs_set_items",
 *   label = @Translation("Breadcrumb: set items"),
 *   type = "breadcrumb"
 * )
 */
class SetBreadcrumbItems extends BreadcrumbActionBase {

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL): void {
    $event = $this->getBreadcrumbEvent();
    if (!$event) {
      return;
    }

    $items_text = $this->configuration['items'] ?? '';
    if (empty($items_text)) {
      return;
    }

    // Prepare token data.
    $this->prepareTokenData($event);

    // Parse and build items.
    $items = $this->parseItems($items_text);
    if (!empty($items)) {
      $event->setItems($items);
    }
  }

  /**
   * Parses items from text configuration.
   *
   * @param string $items_text
   *   The items text (one per line: "Title|URL").
   *
   * @return array
   *   Array of items with 'title' and 'url' keys.
   */
  protected function parseItems(string $items_text): array {
    $lines = array_filter(array_map('trim', explode("\n", $items_text)));
    $items = [];

    foreach ($lines as $line) {
      $parts = array_map('trim', explode('|', $line, 2));
      $title = $parts[0] ?? '';
      $url = $parts[1] ?? '';

      if (empty($title)) {
        continue;
      }

      // Replace tokens.
      $title = $this->tokenService->replaceClear($title);
      $url = $this->tokenService->replaceClear($url);

      $items[] = [
        'title' => $title,
        'url' => $url ?: NULL,
      ];
    }

    return $items;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'items' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['items'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Breadcrumb items'),
      '#description' => $this->t('Enter one item per line in the format: <strong>Title|URL</strong><br>Leave URL empty for current page.<br>Example:<br>Home|/<br>Products|/products<br>[node:title]|'),
      '#default_value' => $this->configuration['items'],
      '#required' => TRUE,
      '#rows' => 10,
      '#weight' => 10,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['items'] = $form_state->getValue('items');
    parent::submitConfigurationForm($form, $form_state);
  }

}
