<?php

namespace Drupal\eca_breadcrumbs\Event;

use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Event dispatched when a breadcrumb is being built.
 */
class BreadcrumbBuildEvent extends Event {

  /**
   * The event name.
   */
  const EVENT_NAME = 'eca_breadcrumbs.build';

  /**
   * The breadcrumb object.
   *
   * @var \Drupal\Core\Breadcrumb\Breadcrumb
   */
  protected Breadcrumb $breadcrumb;

  /**
   * The route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Custom breadcrumb items set by ECA actions.
   *
   * @var array
   */
  protected array $customItems = [];

  /**
   * Constructs a BreadcrumbBuildEvent object.
   *
   * @param \Drupal\Core\Breadcrumb\Breadcrumb $breadcrumb
   *   The breadcrumb object.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   */
  public function __construct(Breadcrumb $breadcrumb, RouteMatchInterface $route_match) {
    $this->breadcrumb = $breadcrumb;
    $this->routeMatch = $route_match;
  }

  /**
   * Gets the breadcrumb object.
   *
   * @return \Drupal\Core\Breadcrumb\Breadcrumb
   *   The breadcrumb object.
   */
  public function getBreadcrumb(): Breadcrumb {
    return $this->breadcrumb;
  }

  /**
   * Gets the route match.
   *
   * @return \Drupal\Core\Routing\RouteMatchInterface
   *   The route match.
   */
  public function getRouteMatch(): RouteMatchInterface {
    return $this->routeMatch;
  }

  /**
   * Adds a custom breadcrumb item.
   *
   * @param string $title
   *   The link title.
   * @param string|null $url
   *   The URL (optional, NULL for current page).
   */
  public function addItem(string $title, ?string $url = NULL): void {
    $this->customItems[] = [
      'title' => $title,
      'url' => $url,
    ];
  }

  /**
   * Sets all breadcrumb items at once.
   *
   * @param array $items
   *   Array of items with 'title' and 'url' keys.
   */
  public function setItems(array $items): void {
    $this->customItems = $items;
  }

  /**
   * Gets the custom breadcrumb items.
   *
   * @return array
   *   The custom items.
   */
  public function getCustomItems(): array {
    return $this->customItems;
  }

  /**
   * Checks if custom items have been set.
   *
   * @return bool
   *   TRUE if custom items exist.
   */
  public function hasCustomItems(): bool {
    return !empty($this->customItems);
  }

}
