<?php

namespace Drupal\eca_breadcrumbs\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca_breadcrumbs\Event\BreadcrumbAppliesEvent;

/**
 * Action to set whether the breadcrumb builder should apply.
 *
 * @Action(
 *   id = "eca_breadcrumbs_set_applies",
 *   label = @Translation("Does breadcrumb should apply"),
 *   description = @Translation("Set whether the breadcrumb builder should apply for this route."),
 *   eca_version_introduced = "1.0.0"
 * )
 *
 * @property array $configuration
 * @property \Drupal\eca\Token\TokenInterface $tokenService
 * @property \Symfony\Contracts\EventDispatcher\Event $event
 */
class SetBreadcrumbApplies extends ConfigurableActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
        'applies' => TRUE,
        'identifier' => '',
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['applies'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Applies'),
      '#description' => $this->t('Check to make the breadcrumb builder apply for this route.'),
      '#default_value' => $this->configuration['applies'],
    ];

    $form['identifier'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pipeline identifier'),
      '#description' => $this->t('Unique identifier for this pipeline. Leave empty to use the ECA model ID automatically. Use the same identifier in the "Build breadcrumb" event condition to ensure only this pipeline builds the breadcrumb. Supports tokens.'),
      '#default_value' => $this->configuration['identifier'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $this->configuration['applies'] = !empty($form_state->getValue('applies'));
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    if (!($this->event instanceof BreadcrumbAppliesEvent)) {
      return;
    }

    $applies = (bool) $this->configuration['applies'];

    $this->event->setApplies($applies);
  }

}
