<?php

namespace Drupal\eca_breadcrumbs;

use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\Breadcrumb\BreadcrumbBuilderInterface;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\Core\Utility\Token;
use Drupal\eca_breadcrumbs\Event\BreadcrumbAppliesEvent;
use Drupal\eca_breadcrumbs\Event\BreadcrumbBuildEvent;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Provides a breadcrumb builder that dispatches ECA events.
 */
class EcaBreadcrumbBuilder implements BreadcrumbBuilderInterface {

  use StringTranslationTrait;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected Token $token;

  /**
   * Cached active identifiers from the last applies() call.
   *
   * @var array|null
   */
  protected ?array $cachedActiveIdentifiers = NULL;

  /**
   * The route match from the last applies() call.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface|null
   */
  protected ?RouteMatchInterface $cachedRouteMatch = NULL;

  /**
   * Constructs an EcaBreadcrumbBuilder object.
   *
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(EventDispatcherInterface $event_dispatcher, Token $token) {
    $this->eventDispatcher = $event_dispatcher;
    $this->token = $token;
  }

  /**
   * {@inheritdoc}
   */
  public function applies(RouteMatchInterface $route_match): bool {
    // Dispatch the ECA event to check if this builder should apply.
    $event = new BreadcrumbAppliesEvent($route_match);
    $this->eventDispatcher->dispatch($event, BreadcrumbAppliesEvent::EVENT_NAME);

    // Cache the active identifiers and route match for build().
    $this->cachedActiveIdentifiers = $event->getActiveIdentifiers();
    $this->cachedRouteMatch = $route_match;

    // If ECA actions set a specific value, use it.
    if ($event->hasAppliesValue()) {
      return $event->getApplies();
    }

    // Default: never apply to prevent extra works on ECA.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function build(RouteMatchInterface $route_match): ?Breadcrumb {
    $breadcrumb = new Breadcrumb();
    $breadcrumb->addCacheContexts(['route']);

    // Reuse cached identifiers if available and route match is the same.
    if ($this->cachedRouteMatch === $route_match && $this->cachedActiveIdentifiers !== NULL) {
      $active_identifiers = $this->cachedActiveIdentifiers;
    }
    else {
      // Cache miss or different route - dispatch applies event.
      $applies_event = new BreadcrumbAppliesEvent($route_match);
      $this->eventDispatcher->dispatch($applies_event, BreadcrumbAppliesEvent::EVENT_NAME);
      $active_identifiers = $applies_event->getActiveIdentifiers();
    }

    // Dispatch the build event with active identifiers.
    $event = new BreadcrumbBuildEvent($breadcrumb, $route_match, $active_identifiers);
    $this->eventDispatcher->dispatch($event, BreadcrumbBuildEvent::EVENT_NAME);

    // If ECA actions set custom items, use them.
    if (!$event->hasCustomItems()) {
      // Return NULL to let other breadcrumb builders handle this route.
      return NULL;
    }

    $links = [];
    foreach ($event->getCustomItems() as $item) {
      $title = $item['title'] ?? '';
      $url = $item['url'] ?? NULL;

      if (empty($title)) {
        continue;
      }

      if ($url === NULL) {
        // Current page - add as text, not link.
        $links[] = Link::createFromRoute($title, '<nolink>');
      }
      else {
        try {
          // Try to create URL.
          if (str_starts_with($url, 'http://') || str_starts_with($url, 'https://')) {
            $url_object = Url::fromUri($url);
          }
          elseif (str_starts_with($url, '/')) {
            $url_object = Url::fromUserInput($url);
          }
          else {
            $url_object = Url::fromRoute($url);
          }

          $links[] = Link::fromTextAndUrl($title, $url_object);
        }
        catch (\Exception $e) {
          // If URL parsing fails, skip this item.
          continue;
        }
      }
    }

    foreach ($links as $link) {
      $breadcrumb->addLink($link);
    }

    return $breadcrumb;
  }

}
