<?php

namespace Drupal\eca_breadcrumbs\Event;

use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Event dispatched when a breadcrumb is being built.
 */
class BreadcrumbBuildEvent extends Event {

  /**
   * The event name.
   */
  const EVENT_NAME = 'eca_breadcrumbs.build';

  /**
   * The breadcrumb object.
   *
   * @var \Drupal\Core\Breadcrumb\Breadcrumb
   */
  protected Breadcrumb $breadcrumb;

  /**
   * The route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Custom breadcrumb items set by ECA actions.
   *
   * @var array
   */
  protected array $customItems = [];

  /**
   * Active pipeline identifiers from the applies event.
   *
   * @var array
   */
  protected array $activeIdentifiers = [];

  /**
   * Constructs a BreadcrumbBuildEvent object.
   *
   * @param \Drupal\Core\Breadcrumb\Breadcrumb $breadcrumb
   *   The breadcrumb object.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param array $active_identifiers
   *   Array of active pipeline identifiers from the applies event.
   */
  public function __construct(Breadcrumb $breadcrumb, RouteMatchInterface $route_match, array $active_identifiers = []) {
    $this->breadcrumb = $breadcrumb;
    $this->routeMatch = $route_match;
    $this->activeIdentifiers = $active_identifiers;
  }

  /**
   * Gets the breadcrumb object.
   *
   * @return \Drupal\Core\Breadcrumb\Breadcrumb
   *   The breadcrumb object.
   */
  public function getBreadcrumb(): Breadcrumb {
    return $this->breadcrumb;
  }

  /**
   * Gets the route match.
   *
   * @return \Drupal\Core\Routing\RouteMatchInterface
   *   The route match.
   */
  public function getRouteMatch(): RouteMatchInterface {
    return $this->routeMatch;
  }

  /**
   * Adds a custom breadcrumb item.
   *
   * @param string $title
   *   The link title.
   * @param string|null $url
   *   The URL (optional, NULL for current page).
   */
  public function addItem(string $title, ?string $url = NULL): void {
    $this->customItems[] = [
      'title' => $title,
      'url' => $url,
    ];
  }

  /**
   * Sets all breadcrumb items at once.
   *
   * @param array $items
   *   Array of items with 'title' and 'url' keys.
   */
  public function setItems(array $items): void {
    $this->customItems = $items;
  }

  /**
   * Gets the custom breadcrumb items.
   *
   * @return array
   *   The custom items.
   */
  public function getCustomItems(): array {
    return $this->customItems;
  }

  /**
   * Checks if custom items have been set.
   *
   * @return bool
   *   TRUE if custom items exist.
   */
  public function hasCustomItems(): bool {
    return !empty($this->customItems);
  }

  /**
   * Gets the active pipeline identifiers.
   *
   * @return array
   *   Array of active identifiers.
   */
  public function getActiveIdentifiers(): array {
    return $this->activeIdentifiers;
  }

  /**
   * Checks if a specific identifier is active.
   *
   * @param string $identifier
   *   The identifier to check.
   *
   * @return bool
   *   TRUE if the identifier is active.
   */
  public function isIdentifierActive(string $identifier): bool {
    return isset($this->activeIdentifiers[$identifier]);
  }

  /**
   * Checks if there are any active identifiers.
   *
   * @return bool
   *   TRUE if at least one identifier is active.
   */
  public function hasActiveIdentifiers(): bool {
    return !empty($this->activeIdentifiers);
  }

}
