<?php

namespace Drupal\eca_breadcrumbs\Plugin\Action;

use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca_breadcrumbs\Event\BreadcrumbBuildEvent;
use Drupal\eca_breadcrumbs\TokenDataHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for breadcrumb actions.
 *
 * @property array $configuration
 * @property \Drupal\eca\Token\TokenInterface $tokenService
 * @property \Symfony\Contracts\EventDispatcher\Event $event
 */
abstract class BreadcrumbActionBase extends ConfigurableActionBase {

  /**
   * The token data helper service.
   *
   * @var \Drupal\eca_breadcrumbs\TokenDataHelper
   */
  protected TokenDataHelper $tokenDataHelper;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->tokenDataHelper = $container->get('eca_breadcrumbs.token_data_helper');
    return $instance;
  }

  /**
   * Gets the breadcrumb event from the current event.
   *
   * @return \Drupal\eca_breadcrumbs\Event\BreadcrumbBuildEvent|null
   *   The breadcrumb event or NULL if not available.
   */
  protected function getBreadcrumbEvent(): ?BreadcrumbBuildEvent {
    return $this->event instanceof BreadcrumbBuildEvent ? $this->event : NULL;
  }

  /**
   * Prepares token data for the given event.
   *
   * @param \Drupal\eca_breadcrumbs\Event\BreadcrumbBuildEvent $event
   *   The breadcrumb event.
   */
  protected function prepareTokenData(BreadcrumbBuildEvent $event): void {
    $this->tokenDataHelper->addRouteParametersToTokenData($event->getRouteMatch());
  }

}
