<?php

namespace Drupal\eca_breadcrumbs\Plugin\ECA\Condition;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\ECA\Condition\ConditionBase;
use Drupal\eca_breadcrumbs\Event\BreadcrumbBuildEvent;

/**
 * Checks if a specific breadcrumb pipeline identifier is active.
 *
 * @EcaCondition(
 *   id = "eca_breadcrumbs_identifier_active",
 *   label = @Translation("Breadcrumb pipeline identifier is active"),
 *   description = @Translation("Checks if a specific pipeline identifier is active in the breadcrumb build event."),
 *   eca_version_introduced = "1.0.0"
 * )
 *
 * @property array $configuration
 * @property \Symfony\Contracts\EventDispatcher\Event $event
 */
class BreadcrumbIdentifierActive extends ConditionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'identifier' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['identifier'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pipeline identifier'),
      '#description' => $this->t('The identifier to check. Must match the identifier from the "Does breadcrumb should apply" action. Supports tokens.'),
      '#default_value' => $this->configuration['identifier'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['identifier'] = $form_state->getValue('identifier', '');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate(): bool {
    if (!($this->event instanceof BreadcrumbBuildEvent)) {
      return FALSE;
    }

    // Support tokens in identifier.
    $identifier = trim((string) $this->tokenService->replaceClear($this->configuration['identifier'] ?? ''));
    if ($identifier === '') {
      // No identifier specified - return FALSE.
      // User must specify the identifier manually in the condition config.
      return FALSE;
    }

    return $this->event->isIdentifierActive($identifier);
  }

}
