<?php

namespace Drupal\eca_breadcrumbs;

use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\Breadcrumb\BreadcrumbBuilderInterface;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\Core\Utility\Token;
use Drupal\eca_breadcrumbs\Event\BreadcrumbBuildEvent;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Provides a breadcrumb builder that dispatches ECA events.
 */
class EcaBreadcrumbBuilder implements BreadcrumbBuilderInterface {

  use StringTranslationTrait;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected Token $token;

  /**
   * The event.
   *
   * @var \Drupal\eca_breadcrumbs\Event\BreadcrumbBuildEvent|null
   */
  protected ?BreadcrumbBuildEvent $event = NULL;

  /**
   * Constructs an EcaBreadcrumbBuilder object.
   *
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(EventDispatcherInterface $event_dispatcher, Token $token) {
    $this->eventDispatcher = $event_dispatcher;
    $this->token = $token;
  }

  /**
   * {@inheritdoc}
   */
  public function applies(RouteMatchInterface $route_match): bool {
    $breadcrumb = new Breadcrumb();
    $breadcrumb->addCacheContexts(['route']);
    $this->event = new BreadcrumbBuildEvent($breadcrumb, $route_match);
    $this->eventDispatcher->dispatch($this->event, BreadcrumbBuildEvent::EVENT_NAME);

    // If ECA actions set custom items, we can apply them.
    return $this->event->hasCustomItems();
  }

  /**
   * {@inheritdoc}
   */
  public function build(RouteMatchInterface $route_match): ?Breadcrumb {
    // If ECA actions set custom items, use them.
    if (!$this->event->hasCustomItems()) {
      // Return NULL to let other breadcrumb builders handle this route.
      return NULL;
    }

    $breadcrumb = $this->event->getBreadcrumb();

    $links = [];
    foreach ($this->event->getCustomItems() as $item) {
      $title = $item['title'] ?? '';
      $url = $item['url'] ?? NULL;

      if (empty($title)) {
        continue;
      }

      if ($url === NULL) {
        // Current page - add as text, not link.
        $links[] = Link::createFromRoute($title, '<nolink>');
      }
      else {
        try {
          // Try to create URL.
          if (str_starts_with($url, 'http://') || str_starts_with($url, 'https://')) {
            $url_object = Url::fromUri($url);
          }
          elseif (str_starts_with($url, '/')) {
            $url_object = Url::fromUserInput($url);
          }
          else {
            $url_object = Url::fromRoute($url);
          }

          $links[] = Link::fromTextAndUrl($title, $url_object);
        }
        catch (\Exception $e) {
          // If URL parsing fails, skip this item.
          continue;
        }
      }
    }

    foreach ($links as $link) {
      $breadcrumb->addLink($link);
    }

    return $breadcrumb;
  }

}
