<?php

namespace Drupal\eca_breadcrumbs\Event;

use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Event dispatched when checking if breadcrumb builder applies.
 */
class BreadcrumbAppliesEvent extends Event {

  /**
   * The event name.
   */
  const EVENT_NAME = 'eca_breadcrumbs.applies';

  /**
   * The route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Whether the breadcrumb builder should apply.
   *
   * @var bool|null
   */
  protected ?bool $applies = NULL;

  /**
   * Identifiers of pipelines that said they apply.
   *
   * @var array
   */
  protected array $activeIdentifiers = [];

  /**
   * Constructs a BreadcrumbAppliesEvent object.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   */
  public function __construct(RouteMatchInterface $route_match) {
    $this->routeMatch = $route_match;
  }

  /**
   * Gets the route match.
   *
   * @return \Drupal\Core\Routing\RouteMatchInterface
   *   The route match.
   */
  public function getRouteMatch(): RouteMatchInterface {
    return $this->routeMatch;
  }

  /**
   * Sets whether the breadcrumb builder should apply.
   *
   * @param bool $applies
   *   TRUE if the builder should apply, FALSE otherwise.
   * @param string|null $identifier
   *   Optional identifier for this pipeline. If applies is TRUE and an
   *   identifier is provided, it will be tracked as an active identifier.
   */
  public function setApplies(bool $applies, ?string $identifier = NULL): void {
    $this->applies = $applies;

    if ($applies && $identifier !== NULL && $identifier !== '') {
      $this->activeIdentifiers[$identifier] = TRUE;
    }
  }

  /**
   * Gets whether the breadcrumb builder should apply.
   *
   * @return bool|null
   *   TRUE if the builder should apply, FALSE if not, NULL if not set.
   */
  public function getApplies(): ?bool {
    return $this->applies;
  }

  /**
   * Checks if the applies value has been set.
   *
   * @return bool
   *   TRUE if applies has been set.
   */
  public function hasAppliesValue(): bool {
    return $this->applies !== NULL;
  }

  /**
   * Gets the active identifiers (pipelines that said they apply).
   *
   * @return array
   *   Array of identifiers keyed by identifier name.
   */
  public function getActiveIdentifiers(): array {
    return $this->activeIdentifiers;
  }

  /**
   * Checks if there are any active identifiers.
   *
   * @return bool
   *   TRUE if at least one identifier is active.
   */
  public function hasActiveIdentifiers(): bool {
    return !empty($this->activeIdentifiers);
  }

}
