<?php

namespace Drupal\eca_drush\Plugin\Action;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

/**
 * Describes the eca_drush execute_drush_command action.
 *
 * @Action(
 *   id = "eca_drush_simple_execute_drush_command",
 *   label = @Translation("Execute Drush command"),
 *   description = @Translation("Runs a specific Drush command with the given
 *   arguments"), eca_version_introduced = "2.1.4" * )
 */
class ExecuteDrushCommandAction extends ConfigurableActionBase {

  /**
   * The absolute path to the Drush executable.
   *
   * @var string
   */
  protected string $drushExecutablePath = '';

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    $access_result = AccessResult::allowed();
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

  /**
   * Prepares environment variables for Drush execution.
   *
   * @return array
   *   An array of environment variables to pass to the Process.
   */
  protected function getDrushEnvironment(): array {
    $environment = [];

    // Set HOME directory if not already set (required by Drush).
    if (empty($_ENV['HOME']) && empty($_SERVER['HOME'])) {
      // Use system temp directory as fallback for HOME.
      $temp_dir = sys_get_temp_dir();
      if (is_dir($temp_dir) && is_writable($temp_dir)) {
        $environment['HOME'] = $temp_dir;
      }
    }
    elseif (!empty($_ENV['HOME'])) {
      $environment['HOME'] = $_ENV['HOME'];
    }
    elseif (!empty($_SERVER['HOME'])) {
      $environment['HOME'] = $_SERVER['HOME'];
    }

    // Set other common environment variables that might be needed.
    if (empty($environment['PATH'])) {
      $path = getenv('PATH');
      if ($path) {
        $environment['PATH'] = $path;
      }
    }

    // Set PHP executable path if available.
    if (defined('PHP_BINARY') && PHP_BINARY) {
      $environment['PHP'] = PHP_BINARY;
    }

    return $environment;
  }

  /**
   * Gets the absolute path to the Drush executable.
   *
   * @return string
   *   The absolute path to drush, or empty string if not found.
   */
  protected function getDrushExecutablePath(): string {
    if (!empty($this->drushExecutablePath) && file_exists($this->drushExecutablePath)) {
      return $this->drushExecutablePath;
    }

    // Try common locations for drush.
    $possible_paths = [
      // Relative to DRUPAL_ROOT (web/).
      DRUPAL_ROOT . '/../vendor/bin/drush',
      // Absolute path from project root.
      dirname(DRUPAL_ROOT) . '/vendor/bin/drush',
      // System-wide drush.
      'drush',
    ];

    foreach ($possible_paths as $path) {
      if ($path === 'drush') {
        // Check if drush is in PATH.
        $which = shell_exec('which drush 2>/dev/null');
        if ($which && file_exists(trim($which))) {
          $this->drushExecutablePath = trim($which);
          return $this->drushExecutablePath;
        }
      }
      elseif (file_exists($path) && is_executable($path)) {
        $this->drushExecutablePath = $path;
        return $this->drushExecutablePath;
      }
    }

    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $drush_path = $this->getDrushExecutablePath();
    if (empty($drush_path)) {
      $this->logger->error('Drush executable not found. Please ensure Drush is installed.');
      return;
    }

    // Command to execute.
    $command = [$drush_path, $this->configuration['command']];

    // Prepare base environment variables for Drush.
    $environment = $this->getDrushEnvironment();
    $environment['ECA_DRUSH_COMMAND_CONTEXT'] = 'running';

    if (!empty($this->configuration['arguments'])) {
      $arguments = $this->configuration['arguments'];
      // Use preg_split to split by spaces not enclosed within quotes.
      $arguments = preg_split('/\s+(?=(?:[^\'"]|\'[^\']*\'|"[^"]*")*$)/', $arguments);
      foreach ($arguments as $argument) {
        if (str_starts_with($argument, '--uri=')) {
          $environment['DRUSH_OPTIONS_URI'] = str_replace('--uri=', '', $argument);
        }
      }
      $command = array_merge($command, $arguments);
    }

    // Set working directory to DRUPAL_ROOT for proper context.
    $working_directory = DRUPAL_ROOT;

    // Initialize the Symfony Process.
    $process = new Process($command, $working_directory, $environment);

    try {
      $process->mustRun();
      $this->logger->info('Drush command executed successfully: @output', [
        '@output' => $process->getOutput(),
      ]);
    }
    catch (ProcessFailedException $e) {
      $this->logger->error('Drush command failed: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'command' => '',
      'arguments' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $drush_path = $this->getDrushExecutablePath();
    if (empty($drush_path)) {
      $form['command'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . $this->t('Drush executable not found. Please ensure Drush is installed in vendor/bin/drush or available in your system PATH.') . '</div>',
      ];
      $form['arguments'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Command Arguments'),
        '#default_value' => $this->configuration['arguments'],
        '#disabled' => TRUE,
      ];
      return parent::buildConfigurationForm($form, $form_state);
    }

    // Command to execute.
    $command = [$drush_path, 'list', '--format=json'];

    // Set working directory to DRUPAL_ROOT for proper context.
    $working_directory = DRUPAL_ROOT;

    // Prepare environment variables for Drush.
    $environment = $this->getDrushEnvironment();

    // Initialize the Symfony Process.
    $process = new Process($command, $working_directory, $environment);

    try {
      $process->mustRun();
      $info = Json::decode($process->getOutput());
    }
    catch (ProcessFailedException $e) {
      $form['command'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . $this->t('Failed to load Drush commands: @error', ['@error' => $e->getMessage()]) . '</div>',
      ];
      $form['arguments'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Command Arguments'),
        '#default_value' => $this->configuration['arguments'],
        '#disabled' => TRUE,
      ];
      return parent::buildConfigurationForm($form, $form_state);
    }

    if (empty($info) || !isset($info['commands'])) {
      $form['command'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . $this->t('Failed to parse Drush command list.') . '</div>',
      ];
      $form['arguments'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Command Arguments'),
        '#default_value' => $this->configuration['arguments'],
        '#disabled' => TRUE,
      ];
      return parent::buildConfigurationForm($form, $form_state);
    }

    $options = [];

    // Group commands by namespace.
    foreach ($info['commands'] as $command) {
      // Skip hidden commands.
      if (!empty($command['hidden'])) {
        continue;
      }

      $namespace = (string) $this->t('Common');
      // Determine the namespace for grouping; default to 'Common'.
      if (isset($info['namespaces'])) {
        foreach ($info['namespaces'] as $namespace_info) {
          if (in_array($command['name'], $namespace_info['commands'], TRUE)) {
            $namespace = (string) $this->t('@namespace (group)', ['@namespace' => $namespace_info['id']]);
            break;
          }
        }
      }

      // Add the command to the appropriate namespace group.
      $options[$namespace][$command['name']] = $this->t('@name (@description)', [
        '@name' => $command['name'],
        '@description' => $command['description'] ?? '',
      ]);
    }

    // Remove empty namespaces/groups.
    $options = array_filter($options, 'count');

    $flat_options = [];
    // Flatten grouping.
    foreach ($options as $namespace => $commands) {
      $flat_options['__namespace__' . $namespace] = $namespace;
      $flat_options += $commands;
    }

    $form['command'] = [
      '#type' => 'select',
      '#required' => TRUE,
      '#title' => $this->t('Drush command'),
      '#options' => ['' => $this->t('- Select -')] + $flat_options,
      '#default_value' => $this->configuration['command'],
      '#description' => $this->t('Application: @name (@version)',
        [
          '@name' => $info['application']['name'] ?? 'Drush',
          '@version' => $info['application']['version'] ?? 'Unknown',
        ]
      ),
    ];
    foreach (array_keys($options) as $namespace) {
      $form['command']['__namespace__' . $namespace]['#disabled'] = TRUE;
      $form['command']['__namespace__' . $namespace]['#attributes']['disabled'] = '';
    }

    // Add initially configured arguments.
    $form['arguments'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Command Arguments'),
      '#default_value' => $this->configuration['arguments'],
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    if (
      ($form_state->getValue('command', '') === '') ||
      str_starts_with($form_state->getValue('command', ''), '__namespace__')
    ) {
      $form_state->setErrorByName('command', $this->t('Please select a command.'));
    }
    parent::validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['command'] = $form_state->getValue('command');
    $this->configuration['arguments'] = $form_state->getValue('arguments');
    parent::submitConfigurationForm($form, $form_state);
  }

}
