<?php

namespace Drupal\eca_entity_import\Plugin\migrate\process;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\eca_migrate\Plugin\migrate\process\Eca;
use Drupal\entity_import\Plugin\migrate\process\EntityImportProcessInterface;
use Drupal\entity_import\Plugin\migrate\process\EntityImportProcessTrait;

/**
 * Define eca entity import.
 *
 * @MigrateProcessPlugin(
 *   id = "eca_entity_import",
 *   label = @Translation("ECA")
 * )
 */
class EcaEntityImport extends Eca implements EntityImportProcessInterface {

  use EntityImportProcessTrait;

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['#markup'] = $this->t('The "@label" process plugin has no configurations. The entity importer ID is "<b>@id</b>". It can be used within the ECA model, e.g. to run a model only on a certain entity import. The entity import ID is available within the ECA model in the token <b>row:source:importer_id</b>.', [
      '@label' => $this->getLabel(),
      '@id' => $this->getImporterId($form_state, $this->messenger()),
    ]);

    return $form;
  }

  /**
   * Returns the importer id.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   *
   * @return string
   *   The importer id.
   */
  protected function getImporterId(FormStateInterface $form_state, MessengerInterface $messenger): string {
    $formObject = $form_state->getFormObject();

    if ($formObject instanceof EntityFormInterface) {
      /** @var \Drupal\entity_import\Entity\EntityImporterFieldMappingInterface $entity */
      $entity = $formObject->getEntity();
      try {
        $id = $entity->getImporterEntity()?->id();
        if ($id) {
          return $id;
        }
      }
      catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
        $messenger->addError($e->getMessage());
      }
    }

    return '';
  }

}
