<?php

namespace Drupal\Tests\eca_entity_import\Unit\Plugin\migrate\process;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\eca\Event\TriggerEvent;
use Drupal\eca_entity_import\Plugin\migrate\process\EcaEntityImport;
use Drupal\entity_import\Entity\EntityImporterFieldMappingInterface;
use Drupal\entity_import\Entity\EntityImporterInterface;
use Drupal\Tests\migrate\Unit\process\MigrateProcessTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for EcaEntityImport migrate process plugin.
 */
#[Group('eca_entity_import')]
#[CoversClass(EcaEntityImport::class)]
class EcaEntityImportTest extends MigrateProcessTestCase {

  /**
   * The entity importer field mapping entity.
   */
  protected EntityImporterFieldMappingInterface|MockObject $entity;

  /**
   * The entity form.
   */
  protected EntityFormInterface|MockObject $formObject;

  /**
   * The form state.
   */
  protected FormStateInterface|MockObject $formState;

  /**
   * The messenger service.
   */
  protected MessengerInterface|MockObject $messenger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $configuration = [];
    $plugin_definition = [
      'id' => 'eca_entity_import',
      'label' => 'ECA',
    ];

    $this->entity = $this->createMock(EntityImporterFieldMappingInterface::class);
    $this->formObject = $this->createMock(EntityFormInterface::class);
    $this->formState = $this->createMock(FormStateInterface::class);
    $this->messenger = $this->createMock(MessengerInterface::class);

    $eventDispatcher = $this->createMock(TriggerEvent::class);
    $this->plugin = new EcaEntityImport($configuration, 'eca_entity_import', $plugin_definition, $eventDispatcher);
  }

  /**
   * Tests buildConfigurationForm method with valid importer ID.
   */
  public function testBuildConfigurationFormWithValidImporterId(): void {
    $importerEntity = $this->createMock(EntityImporterInterface::class);
    $expected_importer_id = 'test_importer';

    $importerEntity->expects($this->once())
      ->method('id')
      ->willReturn($expected_importer_id);

    $this->entity->expects($this->once())
      ->method('getImporterEntity')
      ->willReturn($importerEntity);

    $this->formObject->expects($this->once())
      ->method('getEntity')
      ->willReturn($this->entity);

    $this->formState->expects($this->once())
      ->method('getFormObject')
      ->willReturn($this->formObject);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getImporterId');
    $result = $method->invoke($this->plugin, $this->formState, $this->messenger);
    $this->assertEquals($expected_importer_id, $result);
  }

  /**
   * Tests getImporterId method with InvalidPluginDefinitionException.
   */
  public function testGetImporterIdWithInvalidPluginDefinitionException(): void {
    $expectedExceptionMessage = 'Test exception message';
    $exception = new InvalidPluginDefinitionException('test_plugin', $expectedExceptionMessage);

    $this->entity->expects($this->once())
      ->method('getImporterEntity')
      ->willThrowException($exception);

    $this->formObject->expects($this->once())
      ->method('getEntity')
      ->willReturn($this->entity);

    $this->formState->expects($this->once())
      ->method('getFormObject')
      ->willReturn($this->formObject);

    $this->messenger->expects($this->once())
      ->method('addError')
      ->with($expectedExceptionMessage);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getImporterId');
    $result = $method->invoke($this->plugin, $this->formState, $this->messenger);
    $this->assertEquals('', $result);
  }

}
