<?php

namespace Drupal\Tests\eca_entity_import\Unit;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\eca\Event\TriggerEvent;
use Drupal\eca_entity_import\Plugin\migrate\process\EcaEntityImport;
use Drupal\entity_import\Entity\EntityImporterFieldMappingInterface;
use Drupal\entity_import\Entity\EntityImporterInterface;
use Drupal\Tests\migrate\Unit\process\MigrateProcessTestCase;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests for EcaEntityImport migrate process plugin.
 *
 * @group eca_entity_import
 * @coversDefaultClass \Drupal\eca_entity_import\Plugin\migrate\process\EcaEntityImport
 */
class EcaEntityImportTest extends MigrateProcessTestCase {

  /**
   * The entity importer field mapping entity.
   */
  protected EntityImporterFieldMappingInterface|MockObject $entity;

  /**
   * The entity form.
   */
  protected EntityFormInterface|MockObject $formObject;

  /**
   * The form state.
   */
  protected FormStateInterface|MockObject $formState;

  /**
   * The messenger service.
   */
  protected MessengerInterface|MockObject $messenger;

  /**
   * The importer entity.
   */
  protected EntityImporterInterface|MockObject $importerEntity;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $configuration = [];
    $plugin_definition = [
      'id' => 'eca_entity_import',
      'label' => 'ECA',
    ];

    $this->entity = $this->createMock(EntityImporterFieldMappingInterface::class);
    $this->formObject = $this->createMock(EntityFormInterface::class);
    $this->formState = $this->createMock(FormStateInterface::class);
    $this->messenger = $this->createMock(MessengerInterface::class);
    $this->importerEntity = $this->createMock(EntityImporterInterface::class);

    $this->formObject->expects($this->once())
      ->method('getEntity')
      ->willReturn($this->entity);

    $this->formState->expects($this->once())
      ->method('getFormObject')
      ->willReturn($this->formObject);

    $eventDispatcher = $this->createMock(TriggerEvent::class);
    $this->plugin = new EcaEntityImport($configuration, 'eca_entity_import', $plugin_definition, $eventDispatcher);
  }

  /**
   * Tests buildConfigurationForm method with valid importer ID.
   */
  public function testBuildConfigurationFormWithValidImporterId(): void {
    $form = [];
    $label = 'ECA';
    $expected_importer_id = 'test_importer';
    $expected_markup = 'The "' . $label . '" process plugin has no configurations.';
    $expected_markup .= ' The entity importer ID is "<b>' . $expected_importer_id . '</b>".';
    $expected_markup .= ' It can be used within the ECA model, e.g. to run a model only on a certain entity import.';
    $expected_markup .= ' The entity import ID is available within the ECA model in the token <b>row:source:importer_id</b>.';

    $this->importerEntity->expects($this->once())
      ->method('id')
      ->willReturn($expected_importer_id);

    $this->entity->expects($this->once())
      ->method('getImporterEntity')
      ->willReturn($this->importerEntity);

    $stringTranslation = $this->getStringTranslationStub();
    $this->plugin->setStringTranslation($stringTranslation);

    $result = $this->plugin->buildConfigurationForm($form, $this->formState);
    $this->assertArrayHasKey('#markup', $result);

    $this->assertEquals($expected_markup, $result['#markup']->__toString());
  }

  /**
   * Tests getImporterId method with valid importer ID.
   */
  public function testGetImporterIdWithValidImporterId(): void {
    $expected_importer_id = 'test_importer';

    $this->importerEntity->expects($this->once())
      ->method('id')
      ->willReturn($expected_importer_id);

    $this->entity->expects($this->once())
      ->method('getImporterEntity')
      ->willReturn($this->importerEntity);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getImporterId');
    $result = $method->invoke($this->plugin, $this->formState, $this->messenger);
    $this->assertEquals($expected_importer_id, $result);
  }

  /**
   * Tests getImporterId method with InvalidPluginDefinitionException.
   */
  public function testGetImporterIdWithInvalidPluginDefinitionException(): void {
    $expectedExceptionMessage = 'Test exception message';
    $exception = new InvalidPluginDefinitionException('test_plugin', $expectedExceptionMessage);

    $this->entity->expects($this->once())
      ->method('getImporterEntity')
      ->willThrowException($exception);

    $this->messenger->expects($this->once())
      ->method('addError')
      ->with($expectedExceptionMessage);

    $this->plugin->setMessenger($this->messenger);
    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getImporterId');
    $result = $method->invoke($this->plugin, $this->formState, $this->messenger);
    $this->assertEquals('', $result);
  }

}
