<?php

namespace Drupal\eca_entity_import\Plugin\Action;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca\Service\ContentEntityTypes;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\FieldStorageConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Get the allowed value key of a List field by its label.
 *
 * @Action(
 *   id = "eca_entity_import_get_allowed_value_by_label",
 *   label = @Translation("Get allowed value by label"),
 *   description = @Translation("Get the stored key of a List field option by searching for its label.")
 * )
 */
class GetAllowedValueByLabel extends ConfigurableActionBase {

  /**
   * The list field ids.
   */
  protected const LIST_TEXT_FIELDS = ['list_string', 'list_integer', 'list_float'];

  /**
   * The entity type service.
   */
  protected ContentEntityTypes $entityTypes;


  /**
   * The entity field manager.
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypes = $container->get('eca.service.content_entity_types');
    $instance->entityFieldManager = $container->get('entity_field.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'type' => '',
      'field_name' => '',
      'label' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Entity type (and bundle)'),
      '#options' => $this->entityTypes->getTypesAndBundles(FALSE, FALSE),
      '#description' => $this->t('The entity type and bundle of the List field.'),
      '#default_value' => $this->configuration['type'],
      '#required' => TRUE,
      '#weight' => 10,
      '#eca_token_select_option' => TRUE,
    ];

    $form['field_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Field name'),
      '#description' => $this->t('The machine name of the List field.'),
      '#default_value' => $this->configuration['field_name'],
      '#required' => TRUE,
      '#weight' => 20,
    ];

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label to search'),
      '#description' => $this->t('The label of the option you want to find. Supports tokens.'),
      '#default_value' => $this->configuration['label'],
      '#required' => TRUE,
      '#weight' => 30,
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('The token name to store the found value. Can be used as [eca:token_name].'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
      '#weight' => 40,
      '#eca_token_reference' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['type'] = $form_state->getValue('type');
    $this->configuration['field_name'] = $form_state->getValue('field_name');
    $this->configuration['label'] = $form_state->getValue('label');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    [$entity_type_id, $bundle] = $this->getConfiguredEntityTypeAndBundle();
    $field_name = $this->tokenService->replaceClear($this->configuration['field_name']);
    $label = $this->tokenService->replaceClear($this->configuration['label']);

    if (!$entity_type_id || !$field_name || !$label) {
      return;
    }

    try {
      $field_storage = $this->loadFieldStorageByFieldName($entity_type_id, $field_name);
      if (!$field_storage instanceof FieldStorageConfigInterface
        || !$this->isListField($field_storage->getType(), $field_name)) {
        return;
      }

      $allowed_values = $this->getFieldAllowedValues($field_storage, $field_name);
      if (is_array($allowed_values)) {
        $this->setTokenAllowedValueByLabel($allowed_values, $label, $field_name);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error getting allowed value: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Returns the configured entity type and bundle.
   *
   * @return array
   *   The configured entity type and bundle.
   */
  protected function getConfiguredEntityTypeAndBundle(): array {
    $type = $this->tokenService->replaceClear($this->configuration['type']);
    return explode(' ', $type);
  }

  /**
   * Returns field storage.
   *
   * @param string $entity_type_id
   *   The entity type id of the field.
   * @param string $field_name
   *   The field name.
   *
   * @return \Drupal\field\FieldStorageConfigInterface|null
   *   The field storage config.
   */
  protected function loadFieldStorageByFieldName(string $entity_type_id, string $field_name): ?FieldStorageConfigInterface {
    $field_storage = FieldStorageConfig::loadByName($entity_type_id, $field_name);

    if (!$field_storage) {
      $this->logger->warning('Field storage not found for @entity_type.@field_name', [
        '@entity_type' => $entity_type_id,
        '@field_name' => $field_name,
      ]);
    }

    return $field_storage;
  }

  /**
   * Whether it is a list field.
   *
   * @param string $field_type
   *   The field type.
   * @param string $field_name
   *   The field name.
   *
   * @return bool
   *   Whether it is a list field.
   */
  protected function isListField(string $field_type, string $field_name): bool {
    if (in_array($field_type, static::LIST_TEXT_FIELDS)) {
      return TRUE;
    }

    $this->logger->warning('Field @field_name is not a List field type.', [
      '@field_name' => $field_name,
    ]);

    return FALSE;
  }

  /**
   * Returns allowed values for a field.
   *
   * @param \Drupal\field\FieldStorageConfigInterface $field_storage
   *   The field storage.
   * @param string $field_name
   *   The field_name.
   *
   * @return mixed
   *   The allowed values for the field.
   */
  protected function getFieldAllowedValues(FieldStorageConfigInterface $field_storage, string $field_name): mixed {
    $allowed_values = $field_storage->getSetting('allowed_values');

    if (!$allowed_values) {
      $this->logger->warning('No allowed values found for field @field_name', [
        '@field_name' => $field_name,
      ]);
    }

    return $allowed_values;
  }

  /**
   * Sets token allowed value by label.
   *
   * @param array $allowed_values
   *   The allowed values of a field.
   * @param string $label
   *   The label to return the value for.
   * @param string $field_name
   *   The field name.
   */
  protected function setTokenAllowedValueByLabel(array $allowed_values, string $label, string $field_name): void {
    $key = array_search($label, $allowed_values, TRUE);
    if ($key !== FALSE) {
      $this->tokenService->addTokenData($this->configuration['token_name'], $key);
    }
    else {
      $this->logger->notice('Label "@label" not found in field @field_name allowed values.', [
        '@label' => $label,
        '@field_name' => $field_name,
      ]);
    }
  }

}
