<?php

namespace Drupal\Tests\eca_entity_import\Kernel;

use Drupal\Core\Action\ActionManager;
use Drupal\eca\Plugin\DataType\DataTransferObject;
use Drupal\eca\Token\TokenInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;

/**
 * Kernel tests for the "eca_entity_import_get_allowed_value_by_label" action.
 *
 * @group eca_entity_import
 * @coversDefaultClass \Drupal\eca_entity_import\Plugin\Action\GetAllowedValueByLabel
 */
class GetAllowedValueByLabelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'options',
    'eca',
    'eca_entity_import',
  ];

  /**
   * The action manager.
   */
  protected ActionManager $actionManager;

  /**
   * The token service.
   */
  protected TokenInterface $tokenService;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->actionManager = \Drupal::service('plugin.manager.action');
    $this->tokenService = \Drupal::service('eca.token_services');

    $this->installEntitySchema('node');

    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_status',
      'entity_type' => 'node',
      'type' => 'list_string',
      'settings' => [
        'allowed_values' => [
          'active' => 'Active',
          'inactive' => 'Inactive',
          'pending' => 'Pending Approval',
        ],
      ],
    ]);
    $field_storage->save();

    FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'article',
      'label' => 'Status',
    ])->save();

    $field_storage_int = FieldStorageConfig::create([
      'field_name' => 'field_priority',
      'entity_type' => 'node',
      'type' => 'list_integer',
      'settings' => [
        'allowed_values' => [
          1 => 'Low',
          2 => 'Medium',
          3 => 'High',
        ],
      ],
    ]);
    $field_storage_int->save();

    FieldConfig::create([
      'field_storage' => $field_storage_int,
      'bundle' => 'article',
      'label' => 'Priority',
    ])->save();

    // Create a non-list field for negative testing.
    $field_storage_text = FieldStorageConfig::create([
      'field_name' => 'field_description',
      'entity_type' => 'node',
      'type' => 'text',
    ]);
    $field_storage_text->save();

    FieldConfig::create([
      'field_storage' => $field_storage_text,
      'bundle' => 'article',
      'label' => 'Description',
    ])->save();
  }

  /**
   * Tests with list_string field.
   */
  public function testGetAllowedValueByLabel(): void {
    $config = [
      'type' => 'node article',
      'field_name' => 'field_status',
      'label' => 'Active',
      'token_name' => 'status_value',
    ];
    $action = $this->actionManager->createInstance('eca_entity_import_get_allowed_value_by_label', $config);

    $action->execute();

    $this->assertEquals('active', $this->getTokenValue('status_value'));
  }

  /**
   * Tests with list_integer field.
   */
  public function testGetAllowedValueByLabelInteger(): void {
    $config = [
      'type' => 'node article',
      'field_name' => 'field_priority',
      'label' => 'High',
      'token_name' => 'priority_value',
    ];
    $action = $this->actionManager->createInstance('eca_entity_import_get_allowed_value_by_label', $config);

    $action->execute();

    $this->assertEquals(3, $this->getTokenValue('priority_value'));
  }

  /**
   * Tests with non-existent label.
   */
  public function testGetAllowedValueByLabelNotFound(): void {
    $this->tokenService->addTokenData('status_value', NULL);

    $config = [
      'type' => 'node article',
      'field_name' => 'field_status',
      'label' => 'NonExistentLabel',
      'token_name' => 'status_value',
    ];
    $action = $this->actionManager->createInstance('eca_entity_import_get_allowed_value_by_label', $config);

    $action->execute();

    $this->assertNull($this->getTokenValue('status_value'));
  }

  /**
   * Tests with non-list field.
   */
  public function testGetAllowedValueByLabelNonListField(): void {
    $config = [
      'type' => 'node article',
      'field_name' => 'field_description',
      'label' => 'SomeLabel',
      'token_name' => 'text_value',
    ];
    $action = $this->actionManager->createInstance('eca_entity_import_get_allowed_value_by_label', $config);

    $action->execute();

    $this->assertNull($this->getTokenValue('text_value'));
  }

  /**
   * Tests with non-existent field.
   */
  public function testGetAllowedValueByLabelNonExistentField(): void {
    $this->tokenService->addTokenData('status_value', NULL);

    $config = [
      'type' => 'node article',
      'field_name' => 'field_nonexistent',
      'label' => 'SomeLabel',
      'token_name' => 'missing_value',
    ];
    $action = $this->actionManager->createInstance('eca_entity_import_get_allowed_value_by_label', $config);

    $action->execute();

    $this->assertNull($this->getTokenValue('missing_value'));
  }

  /**
   * Helper to get token value, handling both direct values and DTOs.
   *
   * @param string $token_name
   *   The name of the token to retrieve.
   *
   * @return mixed
   *   The token value.
   */
  protected function getTokenValue(string $token_name): mixed {
    $value = $this->tokenService->getTokenData($token_name);

    return ($value instanceof DataTransferObject)
      ? $value->getValue()
      : $value;
  }

}
