<?php

declare(strict_types=1);

namespace Drupal\eca_external_workflows_pipedream\Plugin\ExternalWorkflowProvider;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca_external_workflows\Attribute\ExternalWorkflowProvider;
use Drupal\eca_external_workflows\ExternalWorkflowProviderPluginBase;

/**
 * Pipedream external workflow provider plugin.
 *
 * Integrates with Pipedream (https://pipedream.com) workflow automation service.
 * Pipedream uses webhook URLs for triggering workflows and supports simple
 * authentication using API keys or tokens from the Key module.
 */
#[ExternalWorkflowProvider(
  id: 'pipedream',
  label: new TranslatableMarkup('Pipedream'),
  description: new TranslatableMarkup('Execute workflows on Pipedream automation platform via webhook URLs.'),
)]
final class PipedreamProvider extends ExternalWorkflowProviderPluginBase {

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function executeWorkflow(string $workflow_id, array $data = [], array $config = [], array $tags = []): array {
    $start_time = microtime(TRUE);

    try {
      // Validate workflow ID (webhook URL)
      if (!$this->validateWorkflowId($workflow_id)) {
        throw new \InvalidArgumentException("Invalid Pipedream workflow URL: {$workflow_id}");
      }

      // Initialize authentication if not already set.
      if ($this->authentication === NULL) {
        try {
          $this->initializeAuthentication();
        } catch (\Exception $auth_error) {
          $this->getLogger()->debug('Authentication initialization failed for Pipedream provider: @error', [
            '@error' => $auth_error->getMessage(),
          ]);
          // Continue without authentication - some webhooks are public
        }
      }

      // Prepare headers with authentication if available.
      $headers = $this->getDefaultHeaders();

      // Add custom headers from config.
      if (isset($config['headers']) && is_array($config['headers'])) {
        $headers = array_merge($headers, $config['headers']);
      }

      // Set up request options.
      $options = [
        'timeout' => $config['timeout'] ?? 30,
        'retry_attempts' => $config['retry_attempts'] ?? 3,
      ];

      // Execute HTTP request via WorkflowHttpClient.
      $result = $this->httpClient->executeRequest(
        $workflow_id,
        $this->getHttpMethod(),
        $data,
        $headers,
        $options,
        array_merge($tags, ['provider:pipedream'])
      );

      // Process result with Pipedream-specific handling.
      return $this->processResponse($result['response_data'] ?? NULL, $result['status_code'] ?? 0);

    }
    catch (\Exception $e) {
      $this->getLogger()->error('Pipedream workflow execution failed: @error', [
        '@error' => $e->getMessage(),
        '@workflow_id' => $workflow_id,
      ]);

      return [
        'success' => FALSE,
        'status_code' => 0,
        'response_data' => NULL,
        'execution_id' => NULL,
        'duration' => round((microtime(TRUE) - $start_time) * 1000),
        'error_message' => $e->getMessage(),
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL): bool {
    // Use base class implementation with authentication loading
    return parent::isUsable($operation_type);
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return ['webhook', 'api'];
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedAuthMethods(): array {
    // Simplified authentication - only single-value credentials from Key module.
    return [
    // Public endpoints (default)
      'none',
    // API key or token authentication.
      'api_key',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableConfiguration(?string $operation_type = NULL): array {
    $config = [
      'timeout' => [
        'type' => 'integer',
        'label' => 'Request timeout (seconds)',
        'default' => 30,
        'min' => 5,
        'max' => 300,
      ],
      'retry_attempts' => [
        'type' => 'integer',
        'label' => 'Maximum retry attempts',
        'default' => 3,
        'min' => 0,
        'max' => 10,
      ],
      'api_key' => [
        'type' => 'key',
        'label' => 'Pipedream API Key',
        'description' => 'Optional API key for authenticated requests',
        'required' => FALSE,
      ],
    ];

    if ($operation_type === 'webhook') {
      $config['webhook_signature'] = [
        'type' => 'string',
        'label' => 'Webhook signature validation',
        'description' => 'Optional webhook signature for security',
        'required' => FALSE,
      ];
    }

    return $config;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultConfigurationValues(?string $operation_type = NULL): array {
    return [
      'timeout' => 30,
      'retry_attempts' => 3,
      'api_key' => '',
      'webhook_signature' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getInputExample(?string $operation_type = NULL): ?array {
    return [
      'name' => 'John Doe',
      'email' => 'john@example.com',
      'action' => 'user_created',
      'timestamp' => date('c'),
      'metadata' => [
        'source' => 'drupal',
        'site' => 'example.com',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthenticationExample(?string $operation_type = NULL): ?array {
    // Simplified authentication examples - only single-value credentials from Key module.
    return [
      'none' => [
        'description' => 'Public webhook - no authentication required',
        'example' => 'Use webhook URL directly without any authentication',
      ],
      'api_key' => [
        'description' => 'API key, token, or secret authentication',
        'example' => 'pipedream_api_key_123456',
        'note' => 'Store your Pipedream API key, token, or secret in the Key module for secure access',
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    // Use base class implementation
    parent::setConfiguration($configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    // Use base class implementation
    return parent::getConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function validateWorkflowId(string $workflow_id, ?string $operation_type = NULL): bool {
    // Pipedream webhook URLs follow the pattern: https://*.m.pipedream.net/*
    return $this->httpClient->validateWebhookUrl($workflow_id, 'pipedream');
  }

  /**
   * {@inheritdoc}
   */
  public function getHttpMethod(?string $operation_type = NULL): string {
    // Pipedream typically uses POST for webhook triggers
    return 'POST';
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultHeaders(?string $operation_type = NULL): array {
    $headers = parent::getDefaultHeaders($operation_type);

    // Add Pipedream-specific headers.
    $headers['User-Agent'] = 'Drupal-ECA-ExternalWorkflows-Pipedream/1.0';

    return $headers;
  }

  /**
   * {@inheritdoc}
   */
  public function processResponse(mixed $response_data, int $status_code): array {
    // Use base class implementation with custom ID/error extraction
    return parent::processResponse($response_data, $status_code);
  }

  /**
   * {@inheritdoc}
   */
  public function setTag(string $tag): void {
    // Use base class implementation
    parent::setTag($tag);
  }

  /**
   * {@inheritdoc}
   */
  public function getTags(): array {
    // Use base class implementation
    return parent::getTags();
  }

  /**
   * {@inheritdoc}
   */
  public function resetTags(): void {
    // Use base class implementation
    parent::resetTags();
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupData(): array {
    return [
      'key_config_name' => 'eca_external_workflows.settings',
      'required_settings' => [],
      'optional_settings' => ['api_key', 'timeout', 'retry_attempts'],
      'operation_types' => [
        'webhook' => [
          'description' => 'Trigger Pipedream workflows via webhook URLs',
          'example_url' => 'https://abc123.m.pipedream.net/webhook-path',
        ],
        'api' => [
          'description' => 'Use Pipedream API for advanced operations',
          'requires_api_key' => TRUE,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function buildAuthenticationHeaders(): array {
    $headers = [];

    if ($this->authentication !== NULL) {
      // Simplified authentication - only handle string tokens from Key module.
      if (is_string($this->authentication)) {
        // Simple API key or token - use Bearer authentication.
        $headers['Authorization'] = 'Bearer ' . $this->authentication;
      }
    }

    return $headers;
  }

  /**
   * {@inheritdoc}
   */
  protected function extractExecutionId(mixed $response_data): ?string {
    if (!is_array($response_data)) {
      return NULL;
    }

    // Pipedream-specific execution ID extraction.
    $id_fields = ['id', 'execution_id', 'run_id', 'request_id'];

    foreach ($id_fields as $field) {
      if (isset($response_data[$field]) && !empty($response_data[$field])) {
        return (string) $response_data[$field];
      }
    }

    // Check nested structures.
    if (isset($response_data['pipedream']) && is_array($response_data['pipedream'])) {
      foreach ($id_fields as $field) {
        if (isset($response_data['pipedream'][$field])) {
          return (string) $response_data['pipedream'][$field];
        }
      }
    }

    return parent::extractExecutionId($response_data);
  }

  /**
   * {@inheritdoc}
   */
  protected function extractErrorMessage(mixed $response_data, int $status_code): ?string {
    if (is_array($response_data)) {
      // Pipedream-specific error message extraction.
      $error_fields = ['error', 'message', 'error_message', 'detail'];

      foreach ($error_fields as $field) {
        if (isset($response_data[$field]) && !empty($response_data[$field])) {
          return (string) $response_data[$field];
        }
      }

      // Check nested error structures.
      if (isset($response_data['error']) && is_array($response_data['error'])) {
        if (isset($response_data['error']['message'])) {
          return (string) $response_data['error']['message'];
        }
      }
    }

    return parent::extractErrorMessage($response_data, $status_code);
  }

  /**
   * {@inheritdoc}
   */
  public function getProviderConfigurationForm(): array {
    // Pipedream is simple - just needs authentication, no advanced settings
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return (string) $this->t('Execute workflows on Pipedream automation platform via webhook URLs. Supports both public webhooks and authenticated API requests.');
  }

}
