<?php

declare(strict_types=1);

namespace Drupal\eca_external_workflows;

use Psr\Log\LoggerInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Base class for external workflow provider plugins.
 *
 * Provides common functionality following AI module patterns including:
 * - Service injection for HTTP client, config, logger, and key repository
 * - Authentication handling with Key module integration
 * - Configuration management with provider-specific defaults
 * - Common HTTP operations and response processing
 * - Debug tag tracking for request monitoring.
 */
abstract class ExternalWorkflowProviderPluginBase extends PluginBase implements ExternalWorkflowProviderInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The workflow HTTP client service.
   */
  protected WorkflowHttpClient $httpClient;

  /**
   * The config factory service.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger factory service.
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * The key repository service.
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * Authentication data for API requests.
   */
  protected mixed $authentication = NULL;

  /**
   * Debug tags for request tracking.
   */
  protected array $tags = [];

  /**
   * Constructs an ExternalWorkflowProviderPluginBase object.
   *
   * @param array $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\eca_external_workflows\WorkflowHttpClient $http_client
   *   The workflow HTTP client service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   */
  final public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    WorkflowHttpClient $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    KeyRepositoryInterface $key_repository,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->keyRepository = $key_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('eca_external_workflows.http_client'),
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('key.repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    // Cast the description to a string since it is a TranslatableMarkup object.
    return (string) ($this->pluginDefinition['description'] ?? '');
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedAuthMethods(): array {
    // Default implementation - most providers support these basic authentication types.
    // Providers should override for specific authentication requirements.
    return ['none', 'api_key'];
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
    $this->authentication = $authentication;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setTag(string $tag): void {
    $this->tags[] = $tag;
  }

  /**
   * {@inheritdoc}
   */
  public function getTags(): array {
    return $this->tags;
  }

  /**
   * {@inheritdoc}
   */
  public function resetTags(): void {
    $this->tags = [];
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL): bool {
    try {
      // Attempt to load authentication if not already set.
      if ($this->authentication === NULL) {
        $this->initializeAuthentication();
      }

      // Base implementation checks if provider has required configuration.
      $setup_data = $this->getSetupData();

      // Check if authentication is required and available.
      $auth_example = $this->getAuthenticationExample($operation_type);
      if ($auth_example !== NULL && $this->authentication === NULL) {
        return FALSE;
      }

      // Check required settings are configured.
      foreach ($setup_data['required_settings'] ?? [] as $setting) {
        if (!isset($this->configuration[$setting]) || empty($this->configuration[$setting])) {
          return FALSE;
        }
      }

      return TRUE;

    }
    catch (\Exception $e) {
      // If authentication loading fails, provider is not usable.
      $this->getLogger()->error('Provider @provider not usable: @error', [
        '@provider' => $this->getPluginId(),
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getHttpMethod(?string $operation_type = NULL): string {
    // Default to POST for workflow execution.
    return 'POST';
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultHeaders(?string $operation_type = NULL): array {
    $headers = [
      'Content-Type' => 'application/json',
      'Accept' => 'application/json',
      'User-Agent' => 'Drupal-ECA-ExternalWorkflows/1.0',
    ];

    // Add authentication headers if available.
    $auth_headers = $this->buildAuthenticationHeaders();
    return array_merge($headers, $auth_headers);
  }

  /**
   * {@inheritdoc}
   */
  public function processResponse(mixed $response_data, int $status_code): array {
    $success = $status_code >= 200 && $status_code < 300;

    return [
      'success' => $success,
      'status_code' => $status_code,
      'response_data' => $response_data,
      'execution_id' => $this->extractExecutionId($response_data),
      'error_message' => $success ? NULL : $this->extractErrorMessage($response_data, $status_code),
    ];
  }

  /**
   * Loads API key from Key module.
   *
   * @param string $key_name
   *   The name of the key to load.
   *
   * @return string|null
   *   The API key value or NULL if not found.
   *
   * @throws \Exception
   *   If the key is not configured or accessible.
   */
  protected function loadApiKey(string $key_name): ?string {
    if (empty($key_name)) {
      return NULL;
    }

    $key = $this->keyRepository->getKey($key_name);
    if (!$key) {
      throw new \Exception("API key '{$key_name}' not found in Key module configuration.");
    }

    $api_key = $key->getKeyValue();
    if (!$api_key) {
      throw new \Exception("API key '{$key_name}' exists but has no value.");
    }

    return $api_key;
  }

  /**
   * Builds authentication headers based on provider configuration.
   *
   * Simplified authentication system supporting only single-value credentials
   * from the Key module (api_key or none).
   *
   * @return array
   *   Array of authentication headers.
   */
  protected function buildAuthenticationHeaders(): array {
    if ($this->authentication === NULL) {
      return [];
    }

    // Default implementation supports only string-based authentication
    // (API keys, tokens, secrets from Key module).
    if (is_string($this->authentication)) {
      // Simple API key/token authentication using Bearer token pattern.
      return ['Authorization' => 'Bearer ' . $this->authentication];
    }

    // Complex authentication structures are not supported.
    // Only simple string tokens from Key module are handled.
    return [];
  }

  /**
   * Extracts execution ID from response data.
   *
   * @param mixed $response_data
   *   Response data from the service.
   *
   * @return string|null
   *   Execution ID if available.
   */
  protected function extractExecutionId(mixed $response_data): ?string {
    // Override in provider implementations to extract specific execution IDs.
    if (is_array($response_data) && isset($response_data['id'])) {
      return (string) $response_data['id'];
    }

    return NULL;
  }

  /**
   * Extracts error message from response data.
   *
   * @param mixed $response_data
   *   Response data from the service.
   * @param int $status_code
   *   HTTP status code.
   *
   * @return string|null
   *   Error message if available.
   */
  protected function extractErrorMessage(mixed $response_data, int $status_code): ?string {
    // Try to extract error from response data.
    if (is_array($response_data)) {
      if (isset($response_data['error'])) {
        return (string) $response_data['error'];
      }
      if (isset($response_data['message'])) {
        return (string) $response_data['message'];
      }
    }

    // Fallback to HTTP status message.
    return "HTTP {$status_code} error occurred";
  }

  /**
   * Gets provider-specific configuration with Key module integration.
   *
   * @param string $config_name
   *   Configuration key name.
   *
   * @return \Drupal\Core\Config\ImmutableConfig
   *   The configuration object.
   */
  protected function getProviderConfig(string $config_name): ImmutableConfig {
    return $this->configFactory->get($config_name);
  }

  /**
   * Gets logger channel for this provider.
   *
   * @return \Psr\Log\LoggerInterface
   *   Logger instance.
   */
  protected function getLogger(): LoggerInterface {
    return $this->loggerFactory->get('eca_external_workflows');
  }

  /**
   * Initializes authentication following AI module patterns.
   *
   * Automatically loads authentication from provider configuration using
   * the Key module integration. Follows the AI module authentication
   * pattern where providers automatically load their auth during startup.
   *
   * @throws \Exception
   *   If authentication is required but cannot be loaded.
   */
  protected function initializeAuthentication(): void {
    // Check if authentication is required for this provider.
    $auth_example = $this->getAuthenticationExample();
    if ($auth_example === NULL) {
      // No authentication required.
      return;
    }

    // Load authentication from centralized configuration.
    $config = $this->configFactory->get('eca_external_workflows.settings');
    $api_key_name = $config->get("providers.{$this->getPluginId()}.api_key");

    if ($api_key_name && $api_key_name !== 'none') {
      // Load API key from Key module and set as authentication.
      $api_key = $this->loadApiKey($api_key_name);
      if ($api_key) {
        $this->setAuthentication($api_key);
        $this->getLogger()->debug('Authentication initialized for provider @provider', [
          '@provider' => $this->getPluginId(),
        ]);
      }
    }
  }

  /**
   * Gets provider configuration object following AI module patterns.
   *
   * Each provider should override this to return their specific config name.
   * This enables automatic authentication loading from provider settings.
   *
   * @return \Drupal\Core\Config\ImmutableConfig|null
   *   Provider configuration or NULL if not configured.
   */
  protected function getConfig(): ?ImmutableConfig {
    $setup_data = $this->getSetupData();
    $config_name = $setup_data['key_config_name'] ?? NULL;

    return $config_name ? $this->getProviderConfig($config_name) : NULL;
  }

}
