<?php

declare(strict_types=1);

namespace Drupal\eca_external_workflows\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca_external_workflows\ExternalWorkflowProviderPluginManager;
use Drupal\eca_external_workflows\Service\AuthenticationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for external workflow providers.
 *
 * Provides a unified interface for configuring all installed external
 * workflow providers in a single form, following the provider plugin pattern.
 */
class ExternalWorkflowsConfigForm extends ConfigFormBase {

  /**
   * The external workflow provider plugin manager.
   */
  protected ExternalWorkflowProviderPluginManager $providerManager;

  /**
   * The authentication manager service.
   */
  protected AuthenticationManager $authenticationManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->providerManager = $container->get('plugin.manager.external_workflow_provider');
    $instance->authenticationManager = $container->get('eca_external_workflows.authentication_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'eca_external_workflows_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['eca_external_workflows.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('eca_external_workflows.settings');

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Configure authentication and settings for external workflow providers.') . '</p>',
    ];

    // Get available providers
    $providers = $this->getEnabledProviders();

    if (empty($providers)) {
      $form['no_providers'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--info">' .
          $this->t('No external workflow providers are installed. Install provider modules like eca_external_workflows_pipedream to get started.') .
          '</div>',
      ];
      return parent::buildForm($form, $form_state);
    }

    // Build unified form with all providers
    foreach ($providers as $provider_id => $provider) {
      $form[$provider_id] = [
        '#type' => 'fieldset',
        '#title' => $provider->label(),
        '#description' => $provider->getDescription(),
      ];

      // Authentication section
      $form[$provider_id]['authentication'] = $this->buildAuthenticationSection($provider_id, $provider, $config);

      // Status section
      $form[$provider_id]['status'] = $this->buildStatusSection($provider, $config);
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Build authentication section for a provider.
   */
  private function buildAuthenticationSection(string $provider_id, $provider, $config): array {
    $current_key = $config->get("providers.{$provider_id}.api_key");
    $auth_options = $this->authenticationManager->getAuthenticationOptions($provider);

    $section = [
      '#type' => 'container',
    ];

    if (count($auth_options) > 1) {
      $section['api_key'] = [
        '#type' => 'select',
        '#title' => $this->t('Authentication'),
        '#options' => $auth_options,
        '#default_value' => $current_key,
        '#description' => $this->t('Select authentication method for @provider', [
          '@provider' => $provider->label(),
        ]),
      ];
    } else {
      // Only one option available
      $option_key = array_key_first($auth_options);
      $section['info'] = [
        '#type' => 'markup',
        '#markup' => '<em>' . $this->t('Authentication: @method', [
          '@method' => $auth_options[$option_key],
        ]) . '</em>',
      ];
      
      // Store the option as hidden value
      $section['api_key'] = [
        '#type' => 'hidden',
        '#value' => $option_key,
      ];
    }

    // Add link to manage keys if api_key authentication is supported
    if (in_array('api_key', $provider->getSupportedAuthMethods())) {
      $key_management_text = $this->authenticationManager->hasAvailableKeys() ?
        $this->t('<a href="@url" target="_blank">Manage API keys</a>', [
          '@url' => $this->authenticationManager->getKeyManagementUrl(),
        ]) :
        $this->t('<a href="@url" target="_blank">Create your first API key</a>', [
          '@url' => $this->authenticationManager->getKeyCreationUrl(),
        ]);

      $section['key_link'] = [
        '#type' => 'markup',
        '#markup' => '<div class="description">' . $key_management_text . '</div>',
      ];
    }

    return $section;
  }

  /**
   * Build status section for a provider.
   */
  private function buildStatusSection($provider, $config): array {
    $provider_id = $provider->getPluginId();
    $current_key = $config->get("providers.{$provider_id}.api_key");
    
    $is_configured = $this->authenticationManager->validateAuthentication($provider, $current_key);

    $status_text = $is_configured ?
      '<span style="color: green;">✓ Ready to use</span>' :
      '<span style="color: orange;">⚠ Needs configuration</span>';

    return [
      '#type' => 'markup',
      '#markup' => '<div class="provider-status"><strong>' . $this->t('Status:') . '</strong> ' . $status_text . '</div>',
    ];
  }

  /**
   * Get enabled/available providers.
   */
  private function getEnabledProviders(): array {
    $providers = [];
    $definitions = $this->providerManager->getDefinitions();

    foreach ($definitions as $provider_id => $definition) {
      try {
        $provider = $this->providerManager->createInstance($provider_id);
        $providers[$provider_id] = $provider;
      } catch (\Exception $e) {
        // Skip providers that can't be instantiated
        continue;
      }
    }

    return $providers;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Validate each provider's authentication
    foreach ($this->getEnabledProviders() as $provider_id => $provider) {
      $api_key = $form_state->getValue([$provider_id, 'authentication', 'api_key']);
      
      if (!$this->authenticationManager->validateAuthentication($provider, $api_key)) {
        $auth_methods = $provider->getSupportedAuthMethods();
        if (!in_array('none', $auth_methods)) {
          $form_state->setErrorByName(
            "{$provider_id}][authentication][api_key",
            $this->t('@provider requires valid authentication to function.', [
              '@provider' => $provider->label(),
            ])
          );
        }
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('eca_external_workflows.settings');

    // Save provider configurations
    foreach ($this->getEnabledProviders() as $provider_id => $provider) {
      $api_key = $form_state->getValue([$provider_id, 'authentication', 'api_key']);
      $config->set("providers.{$provider_id}.api_key", $api_key);
      $config->set("providers.{$provider_id}.enabled", TRUE);
    }

    $config->save();

    $this->messenger()->addMessage($this->t('External workflow provider settings have been saved.'));
    parent::submitForm($form, $form_state);
  }

}
