<?php

declare(strict_types=1);

namespace Drupal\eca_external_workflows;

use Drupal\Component\Plugin\PluginInspectionInterface;

/**
 * Interface for external workflow provider plugins.
 *
 * Defines methods for integrating with external workflow automation services
 * like Pipedream, n8n, Zapier, and other HTTP-based workflow engines.
 */
interface ExternalWorkflowProviderInterface extends PluginInspectionInterface {

  /**
   * Returns the translated plugin label.
   *
   * @return string
   *   The human-readable label for this provider.
   */
  public function label(): string;

  /**
   * Executes a workflow on the external service.
   *
   * @param string $workflow_id
   *   The workflow identifier (webhook URL, workflow ID, etc.).
   * @param array $data
   *   The payload data to send to the workflow.
   * @param array $config
   *   Additional configuration options (headers, timeout, etc.).
   * @param array $tags
   *   Optional tags for tracking and debugging.
   *
   * @return array
   *   Execution result containing:
   *   - success: boolean indicating if execution succeeded
   *   - execution_id: unique identifier for this execution (if available)
   *   - response_data: raw response from the service
   *   - status_code: HTTP status code
   *   - duration: execution time in milliseconds
   *   - error_message: error details if execution failed
   */
  public function executeWorkflow(string $workflow_id, array $data = [], array $config = [], array $tags = []): array;

  /**
   * Checks if the provider is configured and ready to use.
   *
   * @param string|null $operation_type
   *   Optional operation type to check (webhook, api, etc.).
   *
   * @return bool
   *   TRUE if provider is properly configured and can execute workflows.
   */
  public function isUsable(?string $operation_type = NULL): bool;

  /**
   * Returns supported operation types for this provider.
   *
   * @return array
   *   Array of operation types like 'webhook', 'api'.
   */
  public function getSupportedOperationTypes(): array;

  /**
   * Returns available configuration parameters for the provider.
   *
   * @param string|null $operation_type
   *   Optional operation type to get specific configuration for.
   *
   * @return array
   *   Configuration schema with parameter definitions.
   */
  public function getAvailableConfiguration(?string $operation_type = NULL): array;

  /**
   * Returns default configuration values for the provider.
   *
   * @param string|null $operation_type
   *   Optional operation type to get defaults for.
   *
   * @return array
   *   Default configuration values.
   */
  public function getDefaultConfigurationValues(?string $operation_type = NULL): array;

  /**
   * Returns an example of input data structure for workflows.
   *
   * @param string|null $operation_type
   *   Optional operation type to get examples for.
   *
   * @return array|null
   *   Example input structure or NULL if not available.
   */
  public function getInputExample(?string $operation_type = NULL): ?array;

  /**
   * Gets supported authentication methods for this provider.
   *
   * Simplified authentication system using single-value credentials from
   * the Key module. Only supports 'none' or 'api_key' authentication.
   *
   * @return array
   *   Array of supported authentication method IDs:
   *   - 'none': No authentication required (public webhooks)
   *   - 'api_key': API key, token, or secret (single-value from Key module)
   */
  public function getSupportedAuthMethods(): array;

  /**
   * Returns authentication data structure example.
   *
   * @param string|null $operation_type
   *   Optional operation type for authentication.
   *
   * @return array|null
   *   Example authentication structure or NULL if not required.
   */
  public function getAuthenticationExample(?string $operation_type = NULL): ?array;

  /**
   * Sets authentication data for the provider.
   *
   * @param mixed $authentication
   *   Authentication data (API key, token, secret from Key module).
   */
  public function setAuthentication(mixed $authentication): void;

  /**
   * Sets configuration data for the provider.
   *
   * @param array $configuration
   *   Configuration data including timeout, headers, etc.
   */
  public function setConfiguration(array $configuration): void;

  /**
   * Gets current configuration data.
   *
   * @return array
   *   Current configuration data.
   */
  public function getConfiguration(): array;

  /**
   * Validates a workflow identifier format.
   *
   * @param string $workflow_id
   *   The workflow identifier to validate.
   * @param string|null $operation_type
   *   Optional operation type for validation context.
   *
   * @return bool
   *   TRUE if workflow ID is valid for this provider.
   */
  public function validateWorkflowId(string $workflow_id, ?string $operation_type = NULL): bool;

  /**
   * Gets the HTTP method used by this provider.
   *
   * @param string|null $operation_type
   *   Optional operation type to get method for.
   *
   * @return string
   *   HTTP method (GET, POST, PUT, etc.).
   */
  public function getHttpMethod(?string $operation_type = NULL): string;

  /**
   * Gets default headers for requests to this provider.
   *
   * @param string|null $operation_type
   *   Optional operation type for headers.
   *
   * @return array
   *   Array of HTTP headers.
   */
  public function getDefaultHeaders(?string $operation_type = NULL): array;

  /**
   * Processes response data from the external service.
   *
   * @param mixed $response_data
   *   Raw response data from the service.
   * @param int $status_code
   *   HTTP status code.
   *
   * @return array
   *   Processed response with standardized structure.
   */
  public function processResponse(mixed $response_data, int $status_code): array;

  /**
   * Sets a debugging tag for request tracking.
   *
   * @param string $tag
   *   The tag to set.
   */
  public function setTag(string $tag): void;

  /**
   * Gets all debugging tags.
   *
   * @return array
   *   Array of tags set for this provider instance.
   */
  public function getTags(): array;

  /**
   * Clears all debugging tags.
   */
  public function resetTags(): void;

  /**
   * Gets setup data for provider configuration.
   *
   * @return array
   *   Setup information including:
   *   - key_config_name: Key module configuration name
   *   - required_settings: Array of required configuration keys
   *   - optional_settings: Array of optional configuration keys
   *   - operation_types: Supported operation types and their defaults
   */
  public function getSetupData(): array;

  /**
   * Gets provider-specific configuration form elements.
   *
   * These elements will be embedded in the main configuration form
   * for this provider. Should return Form API elements.
   *
   * @return array
   *   Form API elements for provider-specific configuration.
   */
  public function getProviderConfigurationForm(): array;

  /**
   * Gets the description text for this provider.
   *
   * @return string
   *   Human-readable description of what this provider does.
   */
  public function getDescription(): string;

}
