<?php

namespace Drupal\eca_external_workflows\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca_external_workflows\ExternalWorkflowProviderPluginManager;
use Drupal\eca_external_workflows\WorkflowHttpClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Execute workflows on external automation services.
 *
 * This action integrates with external workflow automation platforms
 * like Pipedream, n8n, Zapier, and other HTTP-based workflow engines.
 */
#[Action(
  id: 'eca_external_workflows_execute',
  label: new TranslatableMarkup('External Workflow: Execute'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Execute workflows on external automation services like Pipedream, n8n, Zapier, and other workflow automation platforms.'),
  version_introduced: '1.0.0',
)]
class ExecuteExternalWorkflowAction extends ConfigurableActionBase {

  /**
   * The external workflow provider plugin manager.
   */
  protected ExternalWorkflowProviderPluginManager $providerManager;

  /**
   * The workflow HTTP client service.
   */
  protected WorkflowHttpClient $httpClient;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->providerManager = $container->get('plugin.manager.external_workflow_provider');
    $instance->httpClient = $container->get('eca_external_workflows.http_client');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    // Always allow access during plugin discovery and configuration
    $access_result = AccessResult::allowed();
    return $return_as_object ? $access_result : $access_result->isAllowed();
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $provider_id = $this->configuration['provider'] ?? '';
      $workflow_id = $this->tokenService->replace($this->configuration['workflow_id']);
      $payload_data = $this->configuration['payload_data'] ?? '';
      $result_token = $this->configuration['result_token'] ?? '';

      if (empty($provider_id) || empty($workflow_id) || empty($result_token)) {
        $error_data = [
          'success' => 0,
          'error' => 'Missing required configuration: provider, workflow_id, or result_token',
          'provider' => $provider_id,
          'workflow_id' => $workflow_id,
        ];
        $this->tokenService->addTokenData($result_token ?: 'workflow_error', $error_data);
        return;
      }

      // Create provider instance.
      $provider = $this->providerManager->createInstance($provider_id);

      // Parse payload data.
      $data = [];
      if (!empty($payload_data)) {
        $payload_string = $this->tokenService->replace($payload_data);
        if (!empty($payload_string)) {
          try {
            $data = json_decode($payload_string, TRUE, 512, JSON_THROW_ON_ERROR);
          }
          catch (\JsonException $e) {
            // Try YAML as fallback.
            try {
              $data = Yaml::parse($payload_string) ?: [];
            }
            catch (\Exception $yaml_e) {
              $error_data = [
                'success' => 0,
                'error' => 'Invalid payload format. Must be valid JSON or YAML: ' . $e->getMessage(),
                'provider' => $provider_id,
                'workflow_id' => $workflow_id,
              ];
              $this->tokenService->addTokenData($result_token, $error_data);
              return;
            }
          }
        }
      }

      // Execute workflow.
      $start_time = microtime(TRUE);
      $result = $provider->executeWorkflow($workflow_id, $data, [], ['eca_action']);
      $end_time = microtime(TRUE);

      // Structure response data for ECA.
      $token_data = [
        'success' => $result['success'] ?? 0,
        'provider' => $provider_id,
        'workflow_id' => $workflow_id,
        'execution_id' => $result['execution_id'] ?? NULL,
        'status_code' => $result['status_code'] ?? NULL,
        'response_data' => $result['response_data'] ?? NULL,
        // Milliseconds.
        'duration' => round(($end_time - $start_time) * 1000),
        'timestamp' => time(),
        'error_message' => $result['error_message'] ?? NULL,
      ];

      // Add to ECA token system.
      $this->tokenService->addTokenData($result_token, $token_data);

      // Log execution for debugging.
      $this->logger->info('External workflow executed: @provider/@workflow_id (Success: @success)', [
        '@provider' => $provider_id,
        '@workflow_id' => $workflow_id,
        '@success' => $token_data['success'] ? 'Yes' : 'No',
      ]);

    }
    catch (\Exception $e) {
      // Comprehensive error handling.
      $error_data = [
        'success' => 0,
        'error' => $e->getMessage(),
        'provider' => $provider_id ?? 'unknown',
        'workflow_id' => $workflow_id ?? 'unknown',
        'timestamp' => time(),
      ];

      $result_token = $this->configuration['result_token'] ?? 'workflow_error';
      $this->tokenService->addTokenData($result_token, $error_data);

      $this->logger->error('External workflow execution failed: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'provider' => '',
      'workflow_id' => '',
      'payload_data' => '',
      'result_token' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // IMPORTANT: Call parent first for ECA token browser.
    $form = parent::buildConfigurationForm($form, $form_state);

    // Provider selection dropdown.
    $form['provider'] = [
      '#type' => 'select',
      '#title' => $this->t('Workflow provider'),
      '#description' => $this->t('Select the external workflow service to use.'),
      '#options' => $this->getProviderOptions(),
      '#default_value' => $this->configuration['provider'],
      '#required' => TRUE,
      '#weight' => -50,
    ];

    // Workflow identifier (URL or ID)
    $form['workflow_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Workflow identifier'),
      '#description' => $this->t('The workflow identifier (webhook URL, workflow ID, etc.). Use tokens like [entity:uuid] or [node:title].'),
      '#default_value' => $this->configuration['workflow_id'],
      '#required' => TRUE,
      '#weight' => -40,
      '#eca_token_replacement' => TRUE,
      '#eca_token_reference' => TRUE,
    ];

    // Payload data.
    $form['payload_data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Payload data'),
      '#description' => $this->t('The data to send to the workflow in JSON or YAML format. Use ECA tokens like [entity:field_value]. Example: {"title": "[node:title]", "user": "[current-user:name]"}'),
      '#default_value' => $this->configuration['payload_data'],
      '#rows' => 8,
      '#weight' => -30,
      '#eca_token_replacement' => TRUE,
      '#eca_token_reference' => TRUE,
    ];


    // Result token name.
    $form['result_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result token name'),
      '#description' => $this->t('Token name to store execution results. Use in subsequent actions like [token:workflow_result:success] or [token:workflow_result:response_data].'),
      '#default_value' => $this->configuration['result_token'],
      '#required' => TRUE,
      '#weight' => -10,
      '#eca_token_reference' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);

    // Validate provider selection.
    $provider_id = $form_state->getValue('provider');
    if (!empty($provider_id) && !$this->providerManager->hasDefinition($provider_id)) {
      $form_state->setErrorByName('provider', $this->t('Selected provider is not available.'));
    }

    // Validate token name format.
    $result_token = $form_state->getValue('result_token');
    if (!empty($result_token) && !preg_match('/^[a-zA-Z_][a-zA-Z0-9_]*$/', $result_token)) {
      $form_state->setErrorByName('result_token', $this->t('Token name must start with a letter or underscore and contain only letters, numbers, and underscores.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['provider'] = $form_state->getValue('provider');
    $this->configuration['workflow_id'] = $form_state->getValue('workflow_id');
    $this->configuration['payload_data'] = $form_state->getValue('payload_data');
    $this->configuration['result_token'] = $form_state->getValue('result_token');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * Get available provider options for form dropdown.
   *
   * @return array
   *   Array of provider options keyed by plugin ID.
   */
  protected function getProviderOptions(): array {
    $options = ['' => $this->t('- Select a provider -')];

    $definitions = $this->providerManager->getDefinitions();
    foreach ($definitions as $plugin_id => $definition) {
      $options[$plugin_id] = $definition['label'];
    }

    return $options;
  }

}
