<?php

declare(strict_types=1);

namespace Drupal\eca_external_workflows\Service;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca_external_workflows\ExternalWorkflowProviderInterface;
use Drupal\key\KeyRepositoryInterface;

/**
 * Service for managing external workflow provider authentication.
 *
 * Provides centralized access to authentication methods and Key module integration,
 * allowing the system to potentially swap authentication backends in the future.
 */
class AuthenticationManager {

  use StringTranslationTrait;

  /**
   * The key repository service.
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * Constructs a new AuthenticationManager.
   */
  public function __construct(KeyRepositoryInterface $key_repository) {
    $this->keyRepository = $key_repository;
  }

  /**
   * Get authentication options for a provider.
   *
   * @param \Drupal\eca_external_workflows\ExternalWorkflowProviderInterface $provider
   *   The provider to get authentication options for.
   *
   * @return array
   *   Array of authentication options with keys as option values and values as labels.
   */
  public function getAuthenticationOptions(ExternalWorkflowProviderInterface $provider): array {
    $auth_methods = $provider->getSupportedAuthMethods();
    $options = [];

    // Handle 'none' authentication
    if (in_array('none', $auth_methods)) {
      $options['none'] = $this->t('None (Public access)');
    }

    // Handle 'api_key' authentication
    if (in_array('api_key', $auth_methods)) {
      if (!in_array('none', $auth_methods)) {
        $options[''] = $this->t('- Select API Key -');
      }

      // Get available keys from Key module
      foreach ($this->keyRepository->getKeys() as $key_id => $key) {
        $options[$key_id] = $key->label();
      }
    }

    return $options;
  }

  /**
   * Load authentication value for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   * @param string|null $key_id
   *   The key ID to load, or NULL for no authentication.
   *
   * @return mixed
   *   The authentication value or NULL if not available.
   */
  public function loadAuthentication(string $provider_id, ?string $key_id = NULL): mixed {
    if (empty($key_id) || $key_id === 'none') {
      return NULL; // No authentication
    }

    $key = $this->keyRepository->getKey($key_id);
    return $key ? $key->getKeyValue() : NULL;
  }

  /**
   * Validate authentication for a provider.
   *
   * @param \Drupal\eca_external_workflows\ExternalWorkflowProviderInterface $provider
   *   The provider to validate authentication for.
   * @param string|null $key_id
   *   The key ID to validate, or NULL for no authentication.
   *
   * @return bool
   *   TRUE if authentication is valid for this provider.
   */
  public function validateAuthentication(ExternalWorkflowProviderInterface $provider, ?string $key_id = NULL): bool {
    $auth_methods = $provider->getSupportedAuthMethods();

    // Check if 'none' is valid
    if ((empty($key_id) || $key_id === 'none') && in_array('none', $auth_methods)) {
      return TRUE;
    }

    // Check if key-based authentication is valid
    if (!empty($key_id) && $key_id !== 'none' && in_array('api_key', $auth_methods)) {
      $auth_value = $this->loadAuthentication($provider->getPluginId(), $key_id);
      return !empty($auth_value);
    }

    return FALSE;
  }

  /**
   * Check if any keys are available in the system.
   *
   * @return bool
   *   TRUE if at least one key is available.
   */
  public function hasAvailableKeys(): bool {
    return !empty($this->keyRepository->getKeys());
  }

  /**
   * Get key creation URL for help text.
   *
   * @return string
   *   URL to create new keys.
   */
  public function getKeyCreationUrl(): string {
    return '/admin/config/system/keys/add';
  }

  /**
   * Get key management URL for help text.
   *
   * @return string
   *   URL to manage existing keys.
   */
  public function getKeyManagementUrl(): string {
    return '/admin/config/system/keys';
  }

}
