<?php

declare(strict_types=1);

namespace Drupal\eca_external_workflows;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\TransferException;
use Psr\Http\Message\ResponseInterface;

/**
 * HTTP client service for external workflow execution.
 *
 * Provides centralized HTTP operations for workflow providers including:
 * - Webhook URL execution with retry logic
 * - Authentication header management
 * - Request/response logging and monitoring
 * - Error handling and standardized response processing
 * - Timeout and performance management.
 */
final class WorkflowHttpClient {

  /**
   * Default request timeout in seconds.
   */
  private const DEFAULT_TIMEOUT = 30;

  /**
   * Maximum retry attempts for failed requests.
   */
  private const MAX_RETRY_ATTEMPTS = 3;

  /**
   * Delay between retry attempts in seconds.
   */
  private const RETRY_DELAY = 1;

  /**
   * Constructs a WorkflowHttpClient object.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly LoggerChannelFactoryInterface $loggerFactory,
    private readonly ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Executes a workflow request with retry logic and comprehensive error handling.
   *
   * @param string $url
   *   The webhook URL or API endpoint to call.
   * @param string $method
   *   HTTP method (GET, POST, PUT, PATCH, DELETE).
   * @param array $data
   *   Request payload data.
   * @param array $headers
   *   HTTP headers including authentication.
   * @param array $options
   *   Additional request options (timeout, retry_attempts, etc.).
   * @param array $tags
   *   Debug tags for request tracking.
   *
   * @return array
   *   Standardized response array containing:
   *   - success: boolean indicating if request succeeded
   *   - status_code: HTTP status code
   *   - response_data: decoded response body
   *   - execution_id: unique identifier if available
   *   - duration: request duration in milliseconds
   *   - error_message: error details if request failed
   *   - retry_count: number of retry attempts made
   */
  public function executeRequest(
    string $url,
    string $method = 'POST',
    array $data = [],
    array $headers = [],
    array $options = [],
    array $tags = [],
  ): array {
    $start_time = microtime(TRUE);
    $logger = $this->loggerFactory->get('eca_external_workflows');

    // Validate URL.
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
      return [
        'success' => FALSE,
        'status_code' => 0,
        'response_data' => NULL,
        'execution_id' => NULL,
        'duration' => 0,
        'error_message' => "Invalid URL provided: {$url}",
        'retry_count' => 0,
      ];
    }

    // Prepare request options.
    $request_options = $this->buildRequestOptions($data, $headers, $options);
    $max_attempts = $options['retry_attempts'] ?? self::MAX_RETRY_ATTEMPTS;
    $retry_count = 0;

    // Log request initiation.
    $logger->info('Executing workflow request to @url with method @method', [
      '@url' => $url,
      '@method' => $method,
      '@tags' => implode(', ', $tags),
    ]);

    // Execute request with retry logic.
    for ($attempt = 1; $attempt <= $max_attempts; $attempt++) {
      try {
        $response = $this->httpClient->request($method, $url, $request_options);
        $end_time = microtime(TRUE);
        $duration = (int) round(($end_time - $start_time) * 1000);

        // Process successful response.
        $result = $this->processSuccessfulResponse($response, $duration, $retry_count);

        $logger->info('Workflow request completed successfully in @duration ms (attempt @attempt)', [
          '@duration' => $duration,
          '@attempt' => $attempt,
          '@status' => $result['status_code'],
        ]);

        return $result;

      }
      catch (ConnectException $e) {
        // Connection/timeout errors - retry these.
        $retry_count++;
        $logger->warning('Connection error on attempt @attempt: @error', [
          '@attempt' => $attempt,
          '@error' => $e->getMessage(),
        ]);

        if ($attempt < $max_attempts) {
          // Progressive delay.
          sleep(self::RETRY_DELAY * $attempt);
          continue;
        }

        return $this->buildErrorResponse($e, $start_time, $retry_count);

      }
      catch (RequestException $e) {
        // HTTP errors (4xx, 5xx) - check if we should retry.
        $response = $e->getResponse();
        $status_code = $response ? $response->getStatusCode() : 0;

        $logger->warning('HTTP error @status on attempt @attempt: @error', [
          '@status' => $status_code,
          '@attempt' => $attempt,
          '@error' => $e->getMessage(),
        ]);

        // Retry on 5xx errors but not 4xx errors.
        if ($status_code >= 500 && $attempt < $max_attempts) {
          $retry_count++;
          sleep(self::RETRY_DELAY * $attempt);
          continue;
        }

        // Process error response.
        if ($response) {
          return $this->processErrorResponse($response, $start_time, $retry_count);
        }

        return $this->buildErrorResponse($e, $start_time, $retry_count);

      }
      catch (TransferException $e) {
        // Other transfer errors - usually not retryable.
        $logger->error('Transfer error: @error', ['@error' => $e->getMessage()]);
        return $this->buildErrorResponse($e, $start_time, $retry_count);
      }
    }

    // This should never be reached, but included for completeness.
    return $this->buildErrorResponse(
      new \Exception('Maximum retry attempts reached'),
      $start_time,
      $retry_count
    );
  }

  /**
   * Builds Guzzle request options from parameters.
   *
   * @param array $data
   *   Request payload data.
   * @param array $headers
   *   HTTP headers.
   * @param array $options
   *   Additional options.
   *
   * @return array
   *   Guzzle request options array.
   */
  private function buildRequestOptions(array $data, array $headers, array $options): array {
    $request_options = [
      'timeout' => $options['timeout'] ?? self::DEFAULT_TIMEOUT,
      'connect_timeout' => $options['connect_timeout'] ?? 10,
      'headers' => $headers,
    // We handle HTTP errors manually.
      'http_errors' => FALSE,
    ];

    // Add request body if data provided.
    if (!empty($data)) {
      // Determine content type and encoding.
      $content_type = $headers['Content-Type'] ?? 'application/json';

      if (str_contains($content_type, 'application/json')) {
        $request_options['json'] = $data;
      }
      elseif (str_contains($content_type, 'application/x-www-form-urlencoded')) {
        $request_options['form_params'] = $data;
      }
      else {
        // Default to JSON.
        $request_options['json'] = $data;
      }
    }

    // Add any additional Guzzle options.
    foreach ($options as $key => $value) {
      if (in_array($key, ['verify', 'cert', 'ssl_key', 'proxy', 'allow_redirects'])) {
        $request_options[$key] = $value;
      }
    }

    return $request_options;
  }

  /**
   * Processes a successful HTTP response.
   *
   * @param \Psr\Http\Message\ResponseInterface $response
   *   The HTTP response.
   * @param int $duration
   *   Request duration in milliseconds.
   * @param int $retry_count
   *   Number of retries performed.
   *
   * @return array
   *   Standardized response array.
   */
  private function processSuccessfulResponse(ResponseInterface $response, int $duration, int $retry_count): array {
    $status_code = $response->getStatusCode();
    $body = (string) $response->getBody();

    // Try to decode JSON response.
    $response_data = NULL;
    if (!empty($body)) {
      $decoded = json_decode($body, TRUE);
      $response_data = $decoded ?? $body;
    }

    return [
      'success' => TRUE,
      'status_code' => $status_code,
      'response_data' => $response_data,
      'execution_id' => $this->extractExecutionId($response_data),
      'duration' => $duration,
      'error_message' => NULL,
      'retry_count' => $retry_count,
    ];
  }

  /**
   * Processes an error HTTP response.
   *
   * @param \Psr\Http\Message\ResponseInterface $response
   *   The HTTP response.
   * @param float $start_time
   *   Request start time.
   * @param int $retry_count
   *   Number of retries performed.
   *
   * @return array
   *   Standardized error response array.
   */
  private function processErrorResponse(ResponseInterface $response, float $start_time, int $retry_count): array {
    $end_time = microtime(TRUE);
    $duration = (int) round(($end_time - $start_time) * 1000);
    $status_code = $response->getStatusCode();
    $body = (string) $response->getBody();

    // Try to decode error response.
    $response_data = NULL;
    $error_message = "HTTP {$status_code} error";

    if (!empty($body)) {
      $decoded = json_decode($body, TRUE);
      if ($decoded !== NULL) {
        $response_data = $decoded;
        // Extract error message from various common formats.
        $error_message = $decoded['error'] ?? $decoded['message'] ?? $decoded['error_description'] ?? $error_message;
      }
      else {
        $response_data = $body;
        $error_message = strlen($body) > 100 ? substr($body, 0, 100) . '...' : $body;
      }
    }

    return [
      'success' => FALSE,
      'status_code' => $status_code,
      'response_data' => $response_data,
      'execution_id' => NULL,
      'duration' => $duration,
      'error_message' => $error_message,
      'retry_count' => $retry_count,
    ];
  }

  /**
   * Builds error response from exception.
   *
   * @param \Exception $exception
   *   The exception that occurred.
   * @param float $start_time
   *   Request start time.
   * @param int $retry_count
   *   Number of retries performed.
   *
   * @return array
   *   Standardized error response array.
   */
  private function buildErrorResponse(\Exception $exception, float $start_time, int $retry_count): array {
    $end_time = microtime(TRUE);
    $duration = (int) round(($end_time - $start_time) * 1000);

    return [
      'success' => FALSE,
      'status_code' => 0,
      'response_data' => NULL,
      'execution_id' => NULL,
      'duration' => $duration,
      'error_message' => $exception->getMessage(),
      'retry_count' => $retry_count,
    ];
  }

  /**
   * Extracts execution ID from response data.
   *
   * @param mixed $response_data
   *   Response data from the service.
   *
   * @return string|null
   *   Execution ID if available.
   */
  private function extractExecutionId(mixed $response_data): ?string {
    if (!is_array($response_data)) {
      return NULL;
    }

    // Common execution ID field names across different services.
    $id_fields = ['id', 'execution_id', 'run_id', 'request_id', 'transaction_id'];

    foreach ($id_fields as $field) {
      if (isset($response_data[$field]) && !empty($response_data[$field])) {
        return (string) $response_data[$field];
      }
    }

    return NULL;
  }

  /**
   * Validates webhook URL format for common providers.
   *
   * @param string $url
   *   The URL to validate.
   * @param string|null $provider_type
   *   Optional provider type for specific validation.
   *
   * @return bool
   *   TRUE if URL appears to be a valid webhook URL.
   */
  public function validateWebhookUrl(string $url, ?string $provider_type = NULL): bool {
    // Basic URL validation.
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
      return FALSE;
    }

    // Must use HTTPS for security.
    if (!str_starts_with($url, 'https://')) {
      return FALSE;
    }

    // Provider-specific validation patterns.
    $patterns = [
      'pipedream' => '/^https:\/\/.*\.m\.pipedream\.net(\/.*)?$/',
      'n8n' => '/^https:\/\/.*\/webhook\/.*$/',
      'zapier' => '/^https:\/\/hooks\.zapier\.com\/hooks\/catch\/.*$/',
      'make' => '/^https:\/\/hook\..*\.make\.com\/.*$/',
    ];

    if ($provider_type && isset($patterns[$provider_type])) {
      return preg_match($patterns[$provider_type], $url) === 1;
    }

    // Generic webhook validation - check for common webhook path patterns.
    $webhook_patterns = [
      '/\/webhook/',
      '/\/hook/',
      '/\/trigger/',
      '/\/catch/',
    ];

    foreach ($webhook_patterns as $pattern) {
      if (preg_match($pattern, $url)) {
        return TRUE;
      }
    }

    // If no specific pattern matches, allow any HTTPS URL.
    return TRUE;
  }

}
