<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca\Plugin\FormFieldPluginTrait;
use Drupal\eca_google\GoogleApiService;
use Drupal\eca_google\GoogleAuthActionConfigTrait;
use Drupal\eca_google_sheets\GoogleSheetsActionConfigTrait;
use Drupal\eca_google_sheets\GoogleSheetsService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Append data to a Google Sheet.
 */
#[Action(
  id: 'eca_google_sheets_append_sheet',
  label: new TranslatableMarkup('Google Sheets: Append to Sheet'),
  category: new TranslatableMarkup('Google Sheets'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Appends a single row to a Google Sheet.'),
  version_introduced: '1.0.0',
)]
final class AppendToSheet extends ConfigurableActionBase {

  use FormFieldPluginTrait;
  use GoogleAuthActionConfigTrait;
  use GoogleSheetsActionConfigTrait;

  /**
   * The Google Sheets service.
   */
  protected GoogleSheetsService $googleSheetsService;

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleSheetsService = $container->get('eca_google_sheets.google_sheets');
    $instance->googleApiService = $container->get('eca_google.google_api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'row_data' => '',
      'value_input_option' => GoogleSheetsService::VALUE_INPUT_USER_ENTERED,
    ] + $this->getAuthClientIdDefaultConfig()
      + $this->getSpreadsheetIdDefaultConfig()
      + $this->getSheetRangeDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addSpreadsheetIdConfigurationForm($form, $form_state);
    $form = $this->addSheetRangeConfigurationForm($form, $form_state);

    $form['row_data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Row Data'),
      '#description' => $this->t('Data to append as a single row. Enter one cell value per line. Example:<br/>John Doe<br/>john@example.com<br/>Manager<br/>Supports token replacement.'),
      '#default_value' => $this->configuration['row_data'],
      '#rows' => 5,
      '#eca_token_replacement' => TRUE,
      '#required' => TRUE,
    ];

    $form['value_input_option'] = [
      '#type' => 'select',
      '#title' => $this->t('Value Input Option'),
      '#description' => $this->t('How the input data should be interpreted.'),
      '#options' => $this->googleSheetsService->getValueInputOptions(),
      '#default_value' => $this->configuration['value_input_option'],
      '#required' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->validateApiClientId($form, $form_state);
    $this->validateSheetsAccess($form, $form_state);
    parent::validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $auth_client_id = $this->configuration['auth_client_id'];
    $spreadsheet_id = $this->tokenService->replacePlain($this->configuration['spreadsheet_id']);
    $sheet_range = $this->tokenService->replacePlain($this->configuration['sheet_range']);
    $row_data_text = $this->tokenService->replacePlain($this->configuration['row_data']);
    $value_input_option = $this->configuration['value_input_option'];

    if (empty($auth_client_id) || empty($spreadsheet_id) || empty($row_data_text)) {
      $this->logger->error('Google Sheets append action: Missing required configuration.');
      return;
    }

    // Parse auth_client_id.
    $auth_info = $this->googleApiService->parseAuthClientId($auth_client_id);
    if (!$auth_info) {
      $this->logger->error('AppendToSheet: Invalid auth_client_id format.');
      return;
    }

    $auth_type = $auth_info['auth_type'];
    $client_id = $auth_info['client_id'];

    // Parse row data - one cell per line, creating a single row.
    $lines = array_filter(explode("\n", $row_data_text), 'trim');
    $cells = array_map('trim', $lines);

    if (empty($cells)) {
      $this->logger->warning('Google Sheets append action: No valid cell data found.');
      return;
    }

    // Create a single row with all the cell values.
    $rows = [$cells];

    // Append to Google Sheet.
    $success = $this->googleSheetsService->appendToSheet(
      $auth_type,
      $client_id,
      $spreadsheet_id,
      $sheet_range,
      $rows,
      $value_input_option
    );

    if (!$success) {
      $this->logger->error('Google Sheets append action: Failed to append data to sheet @sheet_id using @auth_type.', [
        '@sheet_id' => $spreadsheet_id,
        '@auth_type' => $auth_type,
      ]);
    }
  }

}
