<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca\Plugin\ECA\PluginFormTrait;
use Drupal\eca_google_sheets\GoogleSheetsService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Action\Attribute\Action;
use Drupal\eca\Attributes\ConfigurablePlugin;
use Drupal\eca\Plugin\Action\Attribute\EcaAction;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca_google\GoogleApiService;
use Drupal\eca_google\GoogleAuthActionConfigTrait;
use Drupal\eca_google_sheets\GoogleSheetsActionConfigTrait;

/**
 * ECA action plugin for clearing data from Google Sheets ranges.
 */
#[Action(
  id: 'eca_google_sheets_clear_sheet',
  label: new TranslatableMarkup('Google Sheets: Clear Sheet'),
  category: new TranslatableMarkup('Google Sheets'),
)]
#[EcaAction(
  description: 'Clear values from specific ranges within Google Sheets while preserving formatting.'
)]
#[ConfigurablePlugin]
class ClearSheet extends ConfigurableActionBase {

  use PluginFormTrait;
  use GoogleAuthActionConfigTrait;
  use GoogleSheetsActionConfigTrait;

  /**
   * The Google Sheets service.
   *
   * @var \Drupal\eca_google_sheets\GoogleSheetsService
   */
  protected GoogleSheetsService $googleSheetsService;

  /**
   * The Google API service.
   *
   * @var \Drupal\eca_google\GoogleApiService
   */
  protected GoogleApiService $googleApiService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleSheetsService = $container->get('eca_google_sheets.google_sheets');
    $instance->googleApiService = $container->get('eca_google.google_api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getAuthClientIdDefaultConfig()
      + $this->getSpreadsheetIdDefaultConfig()
      + $this->getSheetRangeDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addSpreadsheetIdConfigurationForm($form, $form_state);
    $form = $this->addSheetRangeConfigurationForm($form, $form_state);
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->validateApiClientId($form, $form_state);
    $this->validateSheetsAccess($form, $form_state);
    parent::validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $auth_client_id = $this->configuration['auth_client_id'];
    $spreadsheet_id = $this->tokenService->replacePlain($this->configuration['spreadsheet_id']);
    $sheet_range = $this->tokenService->replacePlain($this->configuration['sheet_range']);

    if (empty($auth_client_id) || empty($spreadsheet_id) || empty($sheet_range)) {
      $this->logger->error('ClearSheet: Missing required configuration values.');
      return;
    }

    // Parse auth_client_id.
    $auth_info = $this->googleApiService->parseAuthClientId($auth_client_id);
    if (!$auth_info) {
      $this->logger->error('ClearSheet: Invalid auth_client_id format.');
      return;
    }

    $auth_type = $auth_info['auth_type'];
    $client_id = $auth_info['client_id'];

    // Runtime validation of API access.
    if (!$this->googleSheetsService->validateApiAccess($auth_type, $client_id)) {
      $this->logger->error('ClearSheet: Client @client_id lacks required Sheets API access.', ['@client_id' => $client_id]);
      return;
    }

    // Clear the sheet range values.
    $success = $this->googleSheetsService->clearSheet($auth_type, $client_id, $spreadsheet_id, $sheet_range);
    if (!$success) {
      $this->logger->error('ClearSheet: Failed to clear values from range "@range" in spreadsheet @spreadsheet_id.', [
        '@range' => $sheet_range,
        '@spreadsheet_id' => $spreadsheet_id,
      ]);
      return;
    }
  }

}
