<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca\Plugin\ECA\PluginFormTrait;
use Drupal\eca_google_sheets\GoogleSheetsService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\eca\Plugin\Action\ActionBase;
use Drupal\Core\Action\Attribute\Action;
use Drupal\eca\Attributes\ConfigurablePlugin;
use Drupal\eca\Plugin\Action\Attribute\EcaAction;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca_google\GoogleApiService;
use Drupal\eca_google\GoogleAuthActionConfigTrait;
use Drupal\eca_google_sheets\GoogleSheetsActionConfigTrait;

/**
 * ECA action plugin for deleting entire sheets from Google Spreadsheets.
 */
#[Action(
  id: 'eca_google_sheets_delete_sheet',
  label: new TranslatableMarkup('Google Sheets: Delete Sheet'),
  category: new TranslatableMarkup('Google Sheets'),
)]
#[EcaAction(
  description: 'Delete entire sheets (tabs) from Google Spreadsheets for complete cleanup.'
)]
#[ConfigurablePlugin]
class DeleteSheet extends ConfigurableActionBase {

  use PluginFormTrait;
  use GoogleAuthActionConfigTrait;
  use GoogleSheetsActionConfigTrait;

  /**
   * The Google Sheets service.
   *
   * @var \Drupal\eca_google_sheets\GoogleSheetsService
   */
  protected GoogleSheetsService $googleSheetsService;

  /**
   * The Google API service.
   *
   * @var \Drupal\eca_google\GoogleApiService
   */
  protected GoogleApiService $googleApiService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleSheetsService = $container->get('eca_google_sheets.google_sheets');
    $instance->googleApiService = $container->get('eca_google.google_api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'sheet_identifier' => '',
      'identifier_type' => 'title',
    ] + $this->getAuthClientIdDefaultConfig()
      + $this->getSpreadsheetIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addSpreadsheetIdConfigurationForm($form, $form_state);

    $form['identifier_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Identifier Type'),
      '#description' => $this->t('How to identify the sheet to delete.'),
      '#options' => [
        'title' => $this->t('Sheet Title (name)'),
        'id' => $this->t('Sheet ID (numeric)'),
      ],
      '#default_value' => $this->configuration['identifier_type'],
      '#required' => TRUE,
    ];

    $form['sheet_identifier'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sheet Identifier'),
      '#description' => $this->t('The sheet title (e.g., "TempData") or numeric sheet ID (e.g., from CreateSheet token like [sheet_info:sheetId]). <strong>Warning:</strong> This will permanently delete the entire sheet and all its data.'),
      '#default_value' => $this->configuration['sheet_identifier'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->validateApiClientId($form, $form_state);
    $this->validateSheetsAccess($form, $form_state);
    parent::validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $auth_client_id = $this->configuration['auth_client_id'];
    $spreadsheet_id = $this->tokenService->replacePlain($this->configuration['spreadsheet_id']);
    $sheet_identifier = $this->tokenService->replacePlain($this->configuration['sheet_identifier']);
    $identifier_type = $this->configuration['identifier_type'];

    if (empty($auth_client_id) || empty($spreadsheet_id) || empty($sheet_identifier)) {
      $this->logger->error('DeleteSheet: Missing required configuration values.');
      return;
    }

    // Parse auth_client_id.
    $auth_info = $this->googleApiService->parseAuthClientId($auth_client_id);
    if (!$auth_info) {
      $this->logger->error('DeleteSheet: Invalid auth_client_id format.');
      return;
    }

    $auth_type = $auth_info['auth_type'];
    $client_id = $auth_info['client_id'];

    // Runtime validation of API access.
    if (!$this->googleSheetsService->validateApiAccess($auth_type, $client_id)) {
      $this->logger->error('DeleteSheet: Client @client_id lacks required Sheets API access.', ['@client_id' => $client_id]);
      return;
    }

    // Delete the sheet.
    $success = $this->googleSheetsService->deleteSheet($auth_type, $client_id, $spreadsheet_id, $sheet_identifier, $identifier_type);
    if (!$success) {
      $this->logger->error('DeleteSheet: Failed to delete sheet "@identifier" from spreadsheet @spreadsheet_id.', [
        '@identifier' => $sheet_identifier,
        '@spreadsheet_id' => $spreadsheet_id,
      ]);
      return;
    }
  }

}
