<?php

declare(strict_types=1);

namespace Drupal\eca_google_sheets\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca\Plugin\ECA\PluginFormTrait;
use Drupal\eca_google_sheets\GoogleSheetsService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\eca\Plugin\Action\ActionBase;
use Drupal\Core\Action\Attribute\Action;
use Drupal\eca\Attributes\ConfigurablePlugin;
use Drupal\eca\Plugin\Action\Attribute\EcaAction;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca_google\GoogleApiService;
use Drupal\eca_google\GoogleAuthActionConfigTrait;
use Drupal\eca_google_sheets\GoogleSheetsActionConfigTrait;

/**
 * ECA action plugin for querying Google Sheets data with server-side filtering.
 */
#[Action(
  id: 'eca_google_sheets_query_sheet',
  label: new TranslatableMarkup('Google Sheets: Query Sheet'),
  category: new TranslatableMarkup('Google Sheets'),
)]
#[EcaAction(
  description: 'Query data from Google Sheets using server-side QUERY function with temporary hidden sheet.'
)]
#[ConfigurablePlugin]
class QuerySheet extends ConfigurableActionBase {

  use PluginFormTrait;
  use GoogleAuthActionConfigTrait;
  use GoogleSheetsActionConfigTrait;

  /**
   * The Google Sheets service.
   *
   * @var \Drupal\eca_google_sheets\GoogleSheetsService
   */
  protected GoogleSheetsService $googleSheetsService;

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->googleSheetsService = $container->get('eca_google_sheets.google_sheets');
    $instance->googleApiService = $container->get('eca_google.google_api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'query_expression' => '',
      'token_name' => '',
      'headers_token_name' => '',
      'count_token_name' => '',
      'skip_header_row' => FALSE,
      'use_header_as_keys' => FALSE,
    ] + $this->getAuthClientIdDefaultConfig()
      + $this->getSpreadsheetIdDefaultConfig()
      + $this->getSheetRangeDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);
    $form = $this->addSpreadsheetIdConfigurationForm($form, $form_state);
    $form = $this->addSheetRangeConfigurationForm($form, $form_state);

    $form['query_expression'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Query Expression'),
      '#description' => $this->t('Google Sheets QUERY expression (e.g., "SELECT * WHERE Col1=34 AND Col3>100 ORDER BY Col2"). Use Col1, Col2, etc. to reference columns.'),
      '#default_value' => $this->configuration['query_expression'],
      '#required' => TRUE,
      '#rows' => 3,
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token Name'),
      '#description' => $this->t('Name for the token that will contain the query results. Access data using Google Sheets syntax: [token:Row1:Col1] for first row/column, [token:Row2:Col3] for second row/third column, [token:Row1:_ROW] for row numbers.'),
      '#default_value' => $this->configuration['token_name'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['headers_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Headers Token Name'),
      '#description' => $this->t('Optional token name for the header row data. Leave blank to skip creating this token.'),
      '#default_value' => $this->configuration['headers_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['count_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Count Token Name'),
      '#description' => $this->t('Optional token name for the row count. Leave blank to skip creating this token.'),
      '#default_value' => $this->configuration['count_token_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['skip_header_row'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Skip Header Row'),
      '#description' => $this->t('If checked, the first row will be removed from the query results (useful when query results include column headers).'),
      '#default_value' => $this->configuration['skip_header_row'],
    ];

    $form['use_header_as_keys'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Header Row as Keys'),
      '#description' => $this->t('If checked, the first row will be used as keys for each data row, making it easier to access columns by name (e.g., [query_data:Row1:Name] instead of [query_data:Row1:Col1]).'),
      '#default_value' => $this->configuration['use_header_as_keys'],
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->validateApiClientId($form, $form_state);
    $this->validateSheetsAccess($form, $form_state);
    parent::validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $auth_client_id = $this->configuration['auth_client_id'];
    $spreadsheet_id = $this->tokenService->replacePlain($this->configuration['spreadsheet_id']);
    $sheet_range = $this->tokenService->replacePlain($this->configuration['sheet_range']);
    $query_expression = $this->tokenService->replacePlain($this->configuration['query_expression']);
    $token_name = $this->configuration['token_name'];

    if (empty($auth_client_id) || empty($spreadsheet_id) || empty($sheet_range) || empty($query_expression) || empty($token_name)) {
      $this->logger->error('QuerySheet: Missing required configuration values.');
      return;
    }

    // Parse auth_client_id.
    $auth_info = $this->googleApiService->parseAuthClientId($auth_client_id);
    if (!$auth_info) {
      $this->logger->error('QuerySheet: Invalid auth_client_id format.');
      return;
    }

    $auth_type = $auth_info['auth_type'];
    $client_id = $auth_info['client_id'];

    // Runtime validation of API access.
    if (!$this->googleSheetsService->validateApiAccess($auth_type, $client_id)) {
      $this->logger->error('QuerySheet: Client @client_id lacks required Sheets API access.', ['@client_id' => $client_id]);
      return;
    }

    // Execute the query.
    $data = $this->googleSheetsService->querySheet($auth_type, $client_id, $spreadsheet_id, $sheet_range, $query_expression);
    if ($data === NULL) {
      $this->logger->error('QuerySheet: Failed to query data from Google Sheet.');
      return;
    }

    // Process header row options and row number tracking.
    $skip_header_row = $this->configuration['skip_header_row'];
    $use_header_as_keys = $this->configuration['use_header_as_keys'];
    $processed_data = $data;
    $headers = [];

    if (!empty($data)) {
      if ($skip_header_row || $use_header_as_keys) {
        // Extract the header row since we need to either skip it or use it as keys.
        $headers = array_shift($processed_data);
      }

      if ($use_header_as_keys && !empty($headers)) {
        // Convert each data row to use header values as keys.
        $keyed_data = [];
        foreach ($processed_data as $row) {
          $keyed_row = [];
          foreach ($headers as $index => $header) {
            $keyed_row[$header] = $row[$index] ?? '';
          }
          // Add _ROW key from the last column (row number).
          if (!empty($row)) {
            $keyed_row['_ROW'] = end($row);
          }
          $keyed_data[] = $keyed_row;
        }
        $processed_data = $keyed_data;
      } else {
        // Process _ROW keys for all remaining cases.
        $keyed_data = [];
        foreach ($processed_data as $row_index => $row) {
          if (!$skip_header_row && $row_index === 0) {
            // Keep header row but convert to Google Sheets column syntax.
            $indexed_row = [];
            foreach ($row as $index => $value) {
              $indexed_row['Col' . ($index + 1)] = $value;
            }
            $keyed_data[] = $indexed_row;
            continue;
          }

          // Remove the row number from numeric indices and add as _ROW key.
          if (!empty($row) && is_array($row)) {
            $row_number = array_pop($row);
            // Convert to Google Sheets column syntax for consistency with QUERY
            $indexed_row = [];
            foreach ($row as $index => $value) {
              $indexed_row['Col' . ($index + 1)] = $value;
            }
            $indexed_row['_ROW'] = $row_number;
            $keyed_data[] = $indexed_row;
          } else {
            $keyed_data[] = $row;
          }
        }
        $processed_data = $keyed_data;
      }
    }
    // Convert to Google Sheets row syntax for full consistency.
    $final_data = [];
    $row_index = 1;
    foreach ($processed_data as $row) {
      $final_data['Row' . $row_index] = $row;
      $row_index++;
    }

    // Make the processed data available as ECA tokens.
    $this->tokenService->addTokenData($token_name, $final_data);

    // Add count token if token name is provided.
    $count_token_name = $this->configuration['count_token_name'];
    if (!empty($count_token_name)) {
      $this->tokenService->addTokenData($count_token_name, count($processed_data));
    }

    // Add headers token if headers were extracted and token name is provided.
    $headers_token_name = $this->configuration['headers_token_name'];
    if (!empty($headers) && !empty($headers_token_name)) {
      $this->tokenService->addTokenData($headers_token_name, $headers);
    }
  }

}
