<?php

declare(strict_types=1);

namespace Drupal\eca_google;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\google_api_client\Entity\GoogleApiClient;
use Drupal\google_api_client\Service\GoogleApiClientService;
use Drupal\google_api_client\Service\GoogleApiServiceClientService;
use Drupal\google_api_client\GoogleApiClientInterface;
use Drupal\google_api_client\GoogleApiServiceClientInterface;

/**
 * Service for managing Google API authentication and client operations.
 */
class GoogleApiService {

  use StringTranslationTrait;

  /**
   * The Google API client service (OAuth2).
   */
  protected GoogleApiClientService $googleApiClientService;

  /**
   * The Google API service client service (Service Account).
   */
  protected GoogleApiServiceClientService $googleApiServiceClientService;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * @param GoogleApiClientService $google_api_client_service
   * @param GoogleApiServiceClientService $google_api_service_client_service
   * @param EntityTypeManagerInterface $entity_type_manager
   * @param LoggerChannelFactoryInterface $logger_factory
   */
  public function __construct(
    GoogleApiClientService $google_api_client_service,
    GoogleApiServiceClientService $google_api_service_client_service,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->googleApiClientService = $google_api_client_service;
    $this->googleApiServiceClientService = $google_api_service_client_service;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger_factory->get('eca_google');
  }

  /**
   * Gets a configured Google service instance.
   *
   * @param string $service_name
   *   The service name (e.g., 'sheets', 'drive').
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   *
   * @return object|null
   *   The Google service instance or NULL on failure.
   */
  public function getService(string $service_name, string $auth_type, string $client_id): ?object {
    try {
      if ($auth_type === 'api_client') {
        return $this->getServiceApiClient($service_name, $client_id);
      }
      elseif ($auth_type === 'service_account') {
        return $this->getServiceAccount($service_name, $client_id);
      }
      else {
        $this->logger->error('Invalid authentication type @type. Must be "oauth2" or "service_account".', ['@type' => $auth_type]);
        return NULL;
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to initialize Google @service service: @message', [
        '@service' => $service_name,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Validates that the client has API access for a specific service.
   *
   * @param string $service_name
   *   The service name to validate (e.g., 'sheets', 'drive').
   * @param string $auth_type
   *   The authentication type ('api_client' or 'service_account').
   * @param string $client_id
   *   The Google API client entity ID.
   *
   * @return bool
   *   TRUE if the service API is accessible, FALSE otherwise.
   */
  public function validateApiAccess(string $service_name, string $auth_type, string $client_id): bool {
    $service = $this->getService($service_name, $auth_type, $client_id);
    return $service !== NULL;
  }

  public function getClients(string $auth_type = NULL): array {
    $clients = [];
    if ($auth_type === NULL || $auth_type === 'api_client') {
      $oauth_storage = $this->entityTypeManager->getStorage('google_api_client');
      $clients = array_merge($clients, $oauth_storage->loadMultiple());
    }
    if ($auth_type === NULL || $auth_type === 'service_account') {
      $service_storage = $this->entityTypeManager->getStorage('google_api_service_client');
      $clients = array_merge($clients, $service_storage->loadMultiple());
    }
    return $clients;
  }

  /**
   * Gets all available Google API clients with type prefixes.
   *
   * @return array
   *   Array of client options keyed by 'type:id' format.
   */
  public function getClientOptions(): array {
    $clients = $this->getClients();
    $options = [];
    foreach ($clients as $client) {
      $type = ($client instanceof GoogleApiClient) ? 'api_client' : 'service_account';
      $options[$type . ':' . $client->id()] = $this->t('@client_label (@type_label)', [
        '@client_label' => $client->label(),
        '@type_label' => ($type === 'api_client') ? $this->t('API Client') : $this->t('Service Account'),
      ]);
    }
    return $options;
  }

  /**
   * Parses an auth_client_id string into components.
   *
   * @param string $auth_client_id
   *   The auth client ID in format "auth_type:client_id".
   *
   * @return array|null
   *   Array with 'auth_type' and 'client_id' keys, or NULL if invalid.
   */
  public function parseAuthClientId(string $auth_client_id): ?array {
    if (!str_contains($auth_client_id, ':')) {
      return NULL;
    }

    [$auth_type, $client_id] = explode(':', $auth_client_id, 2);

    if (!in_array($auth_type, ['api_client', 'service_account'])) {
      return NULL;
    }

    return [
      'auth_type' => $auth_type,
      'client_id' => $client_id,
    ];
  }

  /**
   * Gets Google service using OAuth2 authentication.
   *
   * @param string $service_name
   *   The service name (e.g., 'sheets', 'drive').
   * @param string $client_id
   *   The Google API client entity ID.
   *
   * @return object|null
   *   The Google service instance or NULL on failure.
   */
  private function getServiceApiClient(string $service_name, string $client_id): ?object {
    /** @var \Drupal\google_api_client\GoogleApiClientInterface $client */
    $client = $this->entityTypeManager
      ->getStorage('google_api_client')
      ->load($client_id);

    if (!$client instanceof GoogleApiClientInterface) {
      $this->logger->error('Google API client @id not found.', ['@id' => $client_id]);
      return NULL;
    }

    $this->googleApiClientService->setGoogleApiClient($client);
    $service_objects = $this->googleApiClientService->getServiceObjects();

    if (!isset($service_objects[$service_name])) {
      $this->logger->error('@service service not available for API client @id.', [
        '@service' => ucfirst($service_name),
        '@id' => $client_id,
      ]);
      return NULL;
    }

    return $service_objects[$service_name];
  }

  /**
   * Gets Google service using Service Account authentication.
   *
   * @param string $service_name
   *   The service name (e.g., 'sheets', 'drive').
   * @param string $service_client_id
   *   The Google API service client entity ID.
   *
   * @return object|null
   *   The Google service instance or NULL on failure.
   */
  private function getServiceAccount(string $service_name, string $service_client_id): ?object {
    /** @var \Drupal\google_api_client\GoogleApiServiceClientInterface $service_client */
    $service_client = $this->entityTypeManager
      ->getStorage('google_api_service_client')
      ->load($service_client_id);

    if (!$service_client instanceof GoogleApiServiceClientInterface) {
      $this->logger->error('Google API service client @id not found.', ['@id' => $service_client_id]);
      return NULL;
    }

    $this->googleApiServiceClientService->setGoogleApiClient($service_client);
    $service_objects = $this->googleApiServiceClientService->getServiceObjects();

    if (!isset($service_objects[$service_name])) {
      $this->logger->error('@service service not available for service account @id.', [
        '@service' => ucfirst($service_name),
        '@id' => $service_client_id,
      ]);
      return NULL;
    }

    return $service_objects[$service_name];
  }

}
