<?php

declare(strict_types=1);

namespace Drupal\eca_gmail;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca_google\DateTimeFormattingTrait;
use Drupal\eca_google\GoogleApiService;
use Google\Service\Gmail;
use Google\Service\Gmail\Draft;
use Google\Service\Gmail\Message;
use Google\Service\Gmail\VacationSettings;

/**
 * Service for Gmail API operations.
 */
class GmailService {

  use StringTranslationTrait;
  use DateTimeFormattingTrait;

  // Message management actions
  public const string ACTION_ARCHIVE = 'archive';
  public const string ACTION_TRASH = 'trash';
  public const string ACTION_RESTORE = 'restore';
  public const string ACTION_DELETE = 'delete';

  // Message format types
  public const string FORMAT_FULL = 'full';
  public const string FORMAT_METADATA = 'metadata';
  public const string FORMAT_MINIMAL = 'minimal';
  public const string FORMAT_RAW = 'raw';

  // Label operations
  public const string LABEL_ADD = 'add';
  public const string LABEL_REMOVE = 'remove';

  /**
   * The Google API service.
   */
  protected GoogleApiService $googleApiService;

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a new GmailService.
   */
  public function __construct(GoogleApiService $google_api_service, LoggerChannelFactoryInterface $logger_factory) {
    $this->googleApiService = $google_api_service;
    $this->logger = $logger_factory->get('eca_gmail');
  }

  /**
   * Gets available message format options.
   */
  public function getMessageFormatOptions(): array {
    return [
      self::FORMAT_FULL => $this->t('Full (complete message with body and headers)'),
      self::FORMAT_METADATA => $this->t('Metadata (headers and basic info only)'),
      self::FORMAT_MINIMAL => $this->t('Minimal (ID and thread ID only)'),
      self::FORMAT_RAW => $this->t('Raw (RFC 2822 format)'),
    ];
  }

  /**
   * Gets available thread format options (excludes RAW format).
   */
  public function getThreadFormatOptions(): array {
    return [
      self::FORMAT_FULL => $this->t('Full (complete messages with body and headers)'),
      self::FORMAT_METADATA => $this->t('Metadata (headers and basic info only)'),
      self::FORMAT_MINIMAL => $this->t('Minimal (ID and thread ID only)'),
    ];
  }

  /**
   * Gets available message management action options.
   */
  public function getMessageActionOptions(): array {
    return [
      self::ACTION_ARCHIVE => $this->t('Archive (remove from inbox)'),
      self::ACTION_TRASH => $this->t('Move to trash'),
      self::ACTION_RESTORE => $this->t('Restore from trash'),
      self::ACTION_DELETE => $this->t('Delete permanently'),
    ];
  }

  /**
   * Gets available label operation options.
   */
  public function getLabelOperationOptions(): array {
    return [
      self::LABEL_ADD => $this->t('Add labels'),
      self::LABEL_REMOVE => $this->t('Remove labels'),
    ];
  }

  /**
   * Sends an email message.
   */
  public function sendMessage(string $auth_type, string $client_id, array $message_params): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for sending message.');
        return NULL;
      }

      // Create MIME message
      $raw_message = $this->createMimeMessage($message_params);
      if (!$raw_message) {
        $this->logger->error('Failed to create MIME message.');
        return NULL;
      }

      $message = new Message();
      $message->setRaw($raw_message);

      // Set thread ID if replying
      if (!empty($message_params['thread_id'])) {
        $message->setThreadId($message_params['thread_id']);
      }

      $sent_message = $service->users_messages->send('me', $message);

      return $this->formatMessageResult($sent_message);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to send Gmail message: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Searches for messages using Gmail query syntax.
   */
  public function searchMessages(string $auth_type, string $client_id, array $search_params): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for message search.');
        return NULL;
      }


      $options = [
        'maxResults' => $search_params['max_results'] ?? 10,
      ];

      if (!empty($search_params['query'])) {
        $options['q'] = $search_params['query'];
      }

      if (!empty($search_params['page_token'])) {
        $options['pageToken'] = $search_params['page_token'];
      }

      if (!empty($search_params['label_ids'])) {
        $options['labelIds'] = is_array($search_params['label_ids'])
          ? $search_params['label_ids']
          : explode(',', $search_params['label_ids']);
      }


      $messages = $service->users_messages->listUsersMessages('me', $options);

      $results = [
        'messages' => [],
        'total_estimated' => $messages->getResultSizeEstimate(),
        'next_page_token' => $messages->getNextPageToken(),
      ];

      if ($messages->getMessages()) {
        foreach ($messages->getMessages() as $message) {
          $results['messages'][] = [
            'id' => $message->getId(),
            'thread_id' => $message->getThreadId(),
          ];
        }
      }

      return $results;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to search Gmail messages: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves a specific message.
   */
  public function getMessage(string $auth_type, string $client_id, string $message_id, string $format = self::FORMAT_FULL): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for message retrieval.');
        return NULL;
      }

      $message = $service->users_messages->get('me', $message_id, ['format' => $format]);

      return $this->formatMessageResult($message, $format);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get Gmail message: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Sends an existing draft email.
   */
  public function sendDraft(string $auth_type, string $client_id, string $draft_id): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for sending draft.');
        return NULL;
      }

      $draft_request = new \Google\Service\Gmail\Draft();
      $draft_request->setId($draft_id);

      $sent_message = $service->users_drafts->send('me', $draft_request);

      return [
        'id' => $sent_message->getId(),
        'thread_id' => $sent_message->getThreadId(),
        'label_ids' => $sent_message->getLabelIds(),
        'draft_id' => $draft_id,
        'sent_at' => date('Y-m-d H:i:s'),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to send Gmail draft: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Creates a draft message.
   */
  public function createDraft(string $auth_type, string $client_id, array $message_params): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for creating draft.');
        return NULL;
      }

      // Create MIME message
      $raw_message = $this->createMimeMessage($message_params);
      if (!$raw_message) {
        $this->logger->error('Failed to create MIME message for draft.');
        return NULL;
      }

      $message = new Message();
      $message->setRaw($raw_message);

      $draft = new Draft();
      $draft->setMessage($message);

      $created_draft = $service->users_drafts->create('me', $draft);

      return [
        'draft_id' => $created_draft->getId(),
        'message_id' => $created_draft->getMessage()->getId(),
        'thread_id' => $created_draft->getMessage()->getThreadId(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create Gmail draft: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Lists all Gmail labels with their IDs.
   */
  public function listLabels(string $auth_type, string $client_id): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for listing labels.');
        return NULL;
      }

      $labels_response = $service->users_labels->listUsersLabels('me');
      $all_labels = $labels_response->getLabels();

      if (!$all_labels) {
        return [
          'labels' => [],
          'system_labels' => [],
          'user_labels' => [],
          'total_count' => 0,
        ];
      }

      $labels = [];
      $system_labels = [];
      $user_labels = [];

      foreach ($all_labels as $label) {
        $label_data = [
          'id' => $label->getId(),
          'name' => $label->getName(),
          'type' => $label->getType(),
          'messages_total' => $label->getMessagesTotal(),
          'messages_unread' => $label->getMessagesUnread(),
          'threads_total' => $label->getThreadsTotal(),
          'threads_unread' => $label->getThreadsUnread(),
        ];

        $labels[] = $label_data;

        if ($label->getType() === 'system') {
          $system_labels[] = $label_data;
        } else {
          $user_labels[] = $label_data;
        }
      }

      return [
        'labels' => $labels,
        'system_labels' => $system_labels,
        'user_labels' => $user_labels,
        'total_count' => count($labels),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Gmail labels: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Finds a label by its display name and returns full label details.
   */
  public function findLabelByName(string $auth_type, string $client_id, string $label_name): ?array {
    try {
      $labels_result = $this->listLabels($auth_type, $client_id);

      if (!$labels_result || empty($labels_result['labels'])) {
        return NULL;
      }

      foreach ($labels_result['labels'] as $label) {
        if ($label['name'] === $label_name) {
          return $label;
        }
      }

      return NULL;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to find Gmail label by name: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Resolves label identifiers (accepts both IDs and display names).
   */
  public function resolveLabelIds(string $auth_type, string $client_id, array $label_identifiers): array {
    $resolved_ids = [];

    foreach ($label_identifiers as $identifier) {
      $identifier = trim($identifier);

      // Check if it's already a known system label ID
      if (in_array($identifier, ['INBOX', 'STARRED', 'IMPORTANT', 'SENT', 'DRAFT', 'DRAFTS', 'SPAM', 'TRASH', 'UNREAD', 'CHAT'])) {
        $resolved_ids[] = $identifier;
        continue;
      }

      // Check if it looks like a label ID (starts with "Label_" or is a long string)
      if (preg_match('/^(Label_\d+|[a-zA-Z0-9_-]{8,})$/', $identifier)) {
        $resolved_ids[] = $identifier;
        continue;
      }

      // Try to find by display name
      $label_details = $this->findLabelByName($auth_type, $client_id, $identifier);
      if ($label_details && isset($label_details['id'])) {
        $resolved_ids[] = $label_details['id'];
      } else {
        $this->logger->warning('Gmail label not found: @label', ['@label' => $identifier]);
      }
    }

    return $resolved_ids;
  }

  /**
   * Applies labels to messages.
   */
  public function applyLabels(string $auth_type, string $client_id, array $label_params): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for label operations.');
        return NULL;
      }

      $message_ids = is_array($label_params['message_ids'])
        ? $label_params['message_ids']
        : explode(',', $label_params['message_ids']);

      $operation = $label_params['operation'] ?? self::LABEL_ADD;
      $label_ids = is_array($label_params['label_ids'])
        ? $label_params['label_ids']
        : explode(',', $label_params['label_ids']);

      $affected_count = 0;

      foreach ($message_ids as $message_id) {
        $message_id = trim($message_id);
        if (empty($message_id)) continue;

        try {
          $modify_request = new Gmail\ModifyMessageRequest();

          if ($operation === self::LABEL_ADD) {
            $modify_request->setAddLabelIds($label_ids);
          } else {
            $modify_request->setRemoveLabelIds($label_ids);
          }

          $service->users_messages->modify('me', $message_id, $modify_request);
          $affected_count++;
        }
        catch (\Exception $e) {
          $this->logger->warning('Failed to apply labels to message @id: @error', [
            '@id' => $message_id,
            '@error' => $e->getMessage(),
          ]);
        }
      }

      return [
        'operation' => $operation,
        'label_ids' => $label_ids,
        'affected_messages' => $affected_count,
        'total_requested' => count($message_ids),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to apply Gmail labels: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Manages messages (archive, trash, restore, delete).
   */
  public function manageMessages(string $auth_type, string $client_id, array $manage_params): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for message management.');
        return NULL;
      }

      $message_ids = is_array($manage_params['message_ids'])
        ? $manage_params['message_ids']
        : explode(',', $manage_params['message_ids']);

      $action = $manage_params['action'] ?? self::ACTION_ARCHIVE;
      $affected_count = 0;

      foreach ($message_ids as $message_id) {
        $message_id = trim($message_id);
        if (empty($message_id)) continue;

        try {
          switch ($action) {
            case self::ACTION_ARCHIVE:
              $modify_request = new Gmail\ModifyMessageRequest();
              $modify_request->setRemoveLabelIds(['INBOX']);
              $service->users_messages->modify('me', $message_id, $modify_request);
              break;

            case self::ACTION_TRASH:
              $service->users_messages->trash('me', $message_id);
              break;

            case self::ACTION_RESTORE:
              $service->users_messages->untrash('me', $message_id);
              break;

            case self::ACTION_DELETE:
              $service->users_messages->delete('me', $message_id);
              break;
          }
          $affected_count++;
        }
        catch (\Exception $e) {
          $this->logger->warning('Failed to @action message @id: @error', [
            '@action' => $action,
            '@id' => $message_id,
            '@error' => $e->getMessage(),
          ]);
        }
      }

      return [
        'action' => $action,
        'affected_messages' => $affected_count,
        'total_requested' => count($message_ids),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to manage Gmail messages: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Retrieves a thread with all messages.
   */
  public function getThread(string $auth_type, string $client_id, string $thread_id, string $format = self::FORMAT_FULL): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for thread retrieval.');
        return NULL;
      }

      $thread = $service->users_threads->get('me', $thread_id, ['format' => $format]);

      $result = [
        'thread_id' => $thread->getId(),
        'message_count' => 0,
        'messages' => [],
        'snippet' => '',
        'participants' => [],
      ];

      if ($thread->getMessages()) {
        $result['message_count'] = count($thread->getMessages());
        $participants = [];

        foreach ($thread->getMessages() as $message) {
          $formatted_message = $this->formatMessageResult($message, $format);
          $result['messages'][] = $formatted_message;

          // Collect participants
          if (isset($formatted_message['from'])) {
            $participants[] = $formatted_message['from'];
          }
          if (isset($formatted_message['to'])) {
            $participants = array_merge($participants, explode(',', $formatted_message['to']));
          }

          // Use snippet from latest message
          if (isset($formatted_message['snippet'])) {
            $result['snippet'] = $formatted_message['snippet'];
          }
        }

        $result['participants'] = array_values(array_unique(array_map('trim', $participants)));
      }

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get Gmail thread: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Sets vacation auto-reply settings.
   */
  public function setAutoReply(string $auth_type, string $client_id, array $vacation_params): ?array {
    try {
      $service = $this->googleApiService->getService('gmail', $auth_type, $client_id);
      if (!$service instanceof Gmail) {
        $this->logger->error('Failed to get Gmail service for auto-reply settings.');
        return NULL;
      }

      $vacation_settings = new VacationSettings();

      if (isset($vacation_params['enable_auto_reply'])) {
        $vacation_settings->setEnableAutoReply($vacation_params['enable_auto_reply']);
      }

      if (!empty($vacation_params['response_subject'])) {
        $vacation_settings->setResponseSubject($vacation_params['response_subject']);
      }

      if (!empty($vacation_params['response_body_html'])) {
        $vacation_settings->setResponseBodyHtml($vacation_params['response_body_html']);
      }

      if (!empty($vacation_params['response_body_plain_text'])) {
        $vacation_settings->setResponseBodyPlainText($vacation_params['response_body_plain_text']);
      }

      if (!empty($vacation_params['start_time'])) {
        $formatted_start_time = $this->formatDateTimeForApi($vacation_params['start_time']);
        if ($formatted_start_time) {
          // Gmail expects Unix timestamp in milliseconds
          $dt = new \DateTime($formatted_start_time);
          $vacation_settings->setStartTime($dt->getTimestamp() * 1000);
        }
      }

      if (!empty($vacation_params['end_time'])) {
        $formatted_end_time = $this->formatDateTimeForApi($vacation_params['end_time']);
        if ($formatted_end_time) {
          // Gmail expects Unix timestamp in milliseconds
          $dt = new \DateTime($formatted_end_time);
          $vacation_settings->setEndTime($dt->getTimestamp() * 1000);
        }
      }

      if (isset($vacation_params['restrict_to_contacts'])) {
        $vacation_settings->setRestrictToContacts($vacation_params['restrict_to_contacts']);
      }

      if (isset($vacation_params['restrict_to_domain'])) {
        $vacation_settings->setRestrictToDomain($vacation_params['restrict_to_domain']);
      }

      $updated_settings = $service->users_settings->updateVacation('me', $vacation_settings);

      return [
        'enabled' => $updated_settings->getEnableAutoReply(),
        'response_subject' => $updated_settings->getResponseSubject(),
        'response_body_html' => $updated_settings->getResponseBodyHtml(),
        'response_body_plain_text' => $updated_settings->getResponseBodyPlainText(),
        'start_time' => $updated_settings->getStartTime(),
        'end_time' => $updated_settings->getEndTime(),
        'restrict_to_contacts' => $updated_settings->getRestrictToContacts(),
        'restrict_to_domain' => $updated_settings->getRestrictToDomain(),
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to set Gmail auto-reply: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Creates a MIME message from parameters.
   */
  protected function createMimeMessage(array $params): ?string {
    try {
      $to = $params['to'] ?? '';
      $cc = $params['cc'] ?? '';
      $bcc = $params['bcc'] ?? '';
      $subject = $params['subject'] ?? '';
      $body_html = $params['body_html'] ?? '';
      $body_plain = $params['body_plain'] ?? '';
      $reply_to = $params['reply_to'] ?? '';

      if (empty($to) || empty($subject)) {
        return NULL;
      }

      // Build headers
      $headers = [];
      $headers[] = "To: " . $to;

      if (!empty($cc)) {
        $headers[] = "Cc: " . $cc;
      }

      if (!empty($bcc)) {
        $headers[] = "Bcc: " . $bcc;
      }

      if (!empty($reply_to)) {
        $headers[] = "Reply-To: " . $reply_to;
      }

      $headers[] = "Subject: " . $subject;
      $headers[] = "MIME-Version: 1.0";

      // Determine content type
      if (!empty($body_html) && !empty($body_plain)) {
        // Multipart message
        $boundary = uniqid();
        $headers[] = "Content-Type: multipart/alternative; boundary=\"{$boundary}\"";
        $headers[] = "";

        $body = "--{$boundary}\r\n";
        $body .= "Content-Type: text/plain; charset=utf-8\r\n";
        $body .= "Content-Transfer-Encoding: quoted-printable\r\n\r\n";
        $body .= quoted_printable_encode($body_plain) . "\r\n\r\n";

        $body .= "--{$boundary}\r\n";
        $body .= "Content-Type: text/html; charset=utf-8\r\n";
        $body .= "Content-Transfer-Encoding: quoted-printable\r\n\r\n";
        $body .= quoted_printable_encode($body_html) . "\r\n\r\n";

        $body .= "--{$boundary}--";
      }
      elseif (!empty($body_html)) {
        // HTML only
        $headers[] = "Content-Type: text/html; charset=utf-8";
        $headers[] = "Content-Transfer-Encoding: quoted-printable";
        $headers[] = "";
        $body = quoted_printable_encode($body_html);
      }
      else {
        // Plain text
        $headers[] = "Content-Type: text/plain; charset=utf-8";
        $headers[] = "Content-Transfer-Encoding: quoted-printable";
        $headers[] = "";
        $body = quoted_printable_encode($body_plain ?: $subject);
      }

      $message = implode("\r\n", $headers) . "\r\n" . $body;

      return base64_encode($message);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create MIME message: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Formats message result data.
   */
  protected function formatMessageResult(Message $message, string $format = self::FORMAT_FULL): array {
    $result = [
      'id' => $message->getId(),
      'thread_id' => $message->getThreadId(),
    ];

    if ($format === self::FORMAT_MINIMAL) {
      return $result;
    }

    // Add snippet if available
    if ($message->getSnippet()) {
      $result['snippet'] = $message->getSnippet();
    }

    // Add label IDs
    if ($message->getLabelIds()) {
      $result['label_ids'] = $message->getLabelIds();
    }

    if ($format === self::FORMAT_METADATA || $format === self::FORMAT_FULL) {
      $payload = $message->getPayload();
      if ($payload) {
        // Extract headers
        if ($payload->getHeaders()) {
          foreach ($payload->getHeaders() as $header) {
            $name = strtolower($header->getName());
            switch ($name) {
              case 'subject':
                $result['subject'] = $header->getValue();
                break;
              case 'from':
                $result['from'] = $header->getValue();
                break;
              case 'to':
                $result['to'] = $header->getValue();
                break;
              case 'cc':
                $result['cc'] = $header->getValue();
                break;
              case 'bcc':
                $result['bcc'] = $header->getValue();
                break;
              case 'date':
                $result['date'] = $header->getValue();
                break;
              case 'reply-to':
                $result['reply_to'] = $header->getValue();
                break;
            }
          }
        }

        // Extract body for full format
        if ($format === self::FORMAT_FULL) {
          $this->extractMessageBody($payload, $result);
        }
      }
    }

    if ($format === self::FORMAT_RAW && $message->getRaw()) {
      $result['raw'] = $message->getRaw();
    }

    return $result;
  }

  /**
   * Extracts message body from payload.
   */
  protected function extractMessageBody($payload, &$result): void {
    if ($payload->getParts()) {
      // Multipart message
      foreach ($payload->getParts() as $part) {
        $mimeType = $part->getMimeType();
        if ($mimeType === 'text/plain' && !isset($result['body_plain'])) {
          $data = $part->getBody()->getData();
          if ($data) {
            $result['body_plain'] = base64_decode(strtr($data, '-_', '+/'));
          }
        }
        elseif ($mimeType === 'text/html' && !isset($result['body_html'])) {
          $data = $part->getBody()->getData();
          if ($data) {
            $result['body_html'] = base64_decode(strtr($data, '-_', '+/'));
          }
        }
        elseif ($part->getParts()) {
          // Nested multipart
          $this->extractMessageBody($part, $result);
        }
      }
    }
    else {
      // Single part message
      $mimeType = $payload->getMimeType();
      $data = $payload->getBody()->getData();
      if ($data) {
        $body = base64_decode(strtr($data, '-_', '+/'));
        if ($mimeType === 'text/html') {
          $result['body_html'] = $body;
        }
        else {
          $result['body_plain'] = $body;
        }
      }
    }
  }

}
