<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Apply or remove labels to/from Gmail messages.
 */
#[Action(
  id: 'eca_gmail_apply_labels',
  label: new TranslatableMarkup('Gmail: Apply Labels'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Add or remove Gmail labels for email organization with support for batch operations and custom label creation.'),
  version_introduced: '1.0.0',
)]
final class ApplyLabels extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'message_ids' => '',
      'operation' => 'add',
      'label_ids' => '',
      'labels_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['message_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message IDs'),
      '#description' => $this->t('Gmail message IDs to apply labels to (comma-separated). Supports token replacement.'),
      '#default_value' => $this->configuration['message_ids'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['operation'] = [
      '#type' => 'select',
      '#title' => $this->t('Operation'),
      '#description' => $this->t('Whether to add or remove the specified labels.'),
      '#default_value' => $this->configuration['operation'],
      '#options' => $this->gmailService->getLabelOperationOptions(),
      '#required' => TRUE,
    ];

    $form['label_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label IDs'),
      '#description' => $this->t('Label IDs to apply (comma-separated). System labels: INBOX, STARRED, IMPORTANT, SENT, DRAFT, SPAM, TRASH, UNREAD. Custom labels use their Gmail ID. Supports token replacement.'),
      '#default_value' => $this->configuration['label_ids'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['labels_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Operation Results Token Name'),
      '#description' => $this->t('Optional token name to store operation results. Accessible as [token_name:affected_messages], [token_name:operation], [token_name:label_ids], etc.'),
      '#default_value' => $this->configuration['labels_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $message_ids = $this->tokenService->replacePlain($this->configuration['message_ids']);
    $operation = $this->configuration['operation'];
    $label_ids = $this->tokenService->replacePlain($this->configuration['label_ids']);
    $labels_token_name = $this->configuration['labels_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('apply labels');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($message_ids)) {
      $this->logger->error('Gmail apply labels action: Missing message IDs.');
      return;
    }

    if (empty($label_ids)) {
      $this->logger->error('Gmail apply labels action: Missing label IDs.');
      return;
    }

    // Parse message IDs
    $parsed_message_ids = $this->parseMessageIds($message_ids);
    if (empty($parsed_message_ids)) {
      $this->logger->error('Gmail apply labels action: No valid message IDs provided.');
      return;
    }

    // Parse label IDs
    $parsed_label_ids = $this->parseLabelIds($label_ids);
    if (empty($parsed_label_ids)) {
      $this->logger->error('Gmail apply labels action: No valid label IDs provided.');
      return;
    }

    // Prepare label parameters
    $label_params = [
      'message_ids' => $parsed_message_ids,
      'operation' => $operation,
      'label_ids' => $parsed_label_ids,
    ];

    // Apply labels
    $labels_data = $this->gmailService->applyLabels($auth_type, $client_id, $label_params);
    if (!$labels_data) {
      $this->logger->error('Gmail apply labels action: Failed to apply labels using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store labels data in token if requested
    if (!empty($labels_token_name)) {
      $this->tokenService->addTokenData($labels_token_name, $labels_data);
    }
  }

}
