<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Create draft emails in Gmail.
 */
#[Action(
  id: 'eca_gmail_create_draft',
  label: new TranslatableMarkup('Gmail: Create Draft'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Create draft emails for approval workflows or later sending with template support and rich HTML formatting.'),
  version_introduced: '1.0.0',
)]
final class CreateDraft extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'to' => '',
      'cc' => '',
      'bcc' => '',
      'subject' => '',
      'body_html' => '',
      'body_plain' => '',
      'reply_to' => '',
      'draft_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['to'] = [
      '#type' => 'textfield',
      '#title' => $this->t('To'),
      '#description' => $this->t('Recipient email addresses (comma-separated). Supports token replacement.'),
      '#default_value' => $this->configuration['to'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['cc'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CC'),
      '#description' => $this->t('CC email addresses (comma-separated). Supports token replacement.'),
      '#default_value' => $this->configuration['cc'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['bcc'] = [
      '#type' => 'textfield',
      '#title' => $this->t('BCC'),
      '#description' => $this->t('BCC email addresses (comma-separated). Supports token replacement.'),
      '#default_value' => $this->configuration['bcc'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#description' => $this->t('Email subject line. Supports token replacement.'),
      '#default_value' => $this->configuration['subject'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['body_html'] = [
      '#type' => 'textarea',
      '#title' => $this->t('HTML Body'),
      '#description' => $this->t('HTML email content. If both HTML and plain text are provided, a multipart email will be created. Supports token replacement.'),
      '#default_value' => $this->configuration['body_html'],
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    $form['body_plain'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Plain Text Body'),
      '#description' => $this->t('Plain text email content. Used as fallback for HTML emails or as the main content for plain text emails. Supports token replacement.'),
      '#default_value' => $this->configuration['body_plain'],
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    $form['reply_to'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reply-To'),
      '#description' => $this->t('Optional reply-to email address. Supports token replacement.'),
      '#default_value' => $this->configuration['reply_to'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['draft_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Draft Token Name'),
      '#description' => $this->t('Optional token name to store draft information. Accessible as [token_name:draft_id], [token_name:message_id], [token_name:thread_id], etc.'),
      '#default_value' => $this->configuration['draft_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $to = $this->tokenService->replacePlain($this->configuration['to']);
    $cc = $this->tokenService->replacePlain($this->configuration['cc']);
    $bcc = $this->tokenService->replacePlain($this->configuration['bcc']);
    $subject = $this->tokenService->replacePlain($this->configuration['subject']);
    $body_html = $this->tokenService->replace($this->configuration['body_html']);
    $body_plain = $this->tokenService->replace($this->configuration['body_plain']);
    $reply_to = $this->tokenService->replacePlain($this->configuration['reply_to']);
    $draft_token_name = $this->configuration['draft_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('create draft');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($to) || empty($subject)) {
      $this->logger->error('Gmail create draft action: Missing required fields (to, subject).');
      return;
    }

    // Validate email addresses
    $to_addresses = $this->parseEmailAddresses($to);
    foreach ($to_addresses as $email) {
      if (!$this->validateEmailAddress($email)) {
        $this->logger->error('Gmail create draft action: Invalid email address in To field: @email', [
          '@email' => $email,
        ]);
        return;
      }
    }

    if (!empty($cc)) {
      $cc_addresses = $this->parseEmailAddresses($cc);
      foreach ($cc_addresses as $email) {
        if (!$this->validateEmailAddress($email)) {
          $this->logger->error('Gmail create draft action: Invalid email address in CC field: @email', [
            '@email' => $email,
          ]);
          return;
        }
      }
    }

    if (!empty($bcc)) {
      $bcc_addresses = $this->parseEmailAddresses($bcc);
      foreach ($bcc_addresses as $email) {
        if (!$this->validateEmailAddress($email)) {
          $this->logger->error('Gmail create draft action: Invalid email address in BCC field: @email', [
            '@email' => $email,
          ]);
          return;
        }
      }
    }

    if (empty($body_html) && empty($body_plain)) {
      $this->logger->error('Gmail create draft action: Either HTML body or plain text body must be provided.');
      return;
    }


    // Prepare message parameters
    $message_params = [
      'to' => $to,
      'subject' => $subject,
    ];

    if (!empty($cc)) {
      $message_params['cc'] = $cc;
    }

    if (!empty($bcc)) {
      $message_params['bcc'] = $bcc;
    }

    if (!empty($body_html)) {
      $message_params['body_html'] = $body_html;
    }

    if (!empty($body_plain)) {
      $message_params['body_plain'] = $body_plain;
    }

    if (!empty($reply_to)) {
      $message_params['reply_to'] = $reply_to;
    }

    // Create the draft
    $draft_data = $this->gmailService->createDraft($auth_type, $client_id, $message_params);
    if (!$draft_data) {
      $this->logger->error('Gmail create draft action: Failed to create draft using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store draft data in token if requested
    if (!empty($draft_token_name)) {
      $this->tokenService->addTokenData($draft_token_name, $draft_data);
    }
  }

}
