<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Find a Gmail label ID by its display name.
 */
#[Action(
  id: 'eca_gmail_get_label_by_name',
  label: new TranslatableMarkup('Gmail: Get Label by Name'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Find the internal Gmail label ID for a given display name, useful for preparing label IDs for Apply Labels action.'),
  version_introduced: '1.0.0',
)]
final class GetLabelByName extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'label_name' => '',
      'label_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['label_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label Display Name'),
      '#description' => $this->t('The display name of the Gmail label to look up (e.g., "Test", "Important", "My Custom Label"). Supports token replacement.'),
      '#default_value' => $this->configuration['label_name'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['label_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label Token Name'),
      '#description' => $this->t('Optional token name to store the found label information. Accessible as [token_name:id], [token_name:name], [token_name:type], etc.'),
      '#default_value' => $this->configuration['label_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $label_name = $this->tokenService->replacePlain($this->configuration['label_name']);
    $label_token_name = $this->configuration['label_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get label by name');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($label_name)) {
      $this->logger->error('Gmail get label by name action: No label name provided.');
      return;
    }

    // Find label by name (returns full details in one call)
    $label_details = $this->gmailService->findLabelByName($auth_type, $client_id, $label_name);

    if (!$label_details) {
      $this->logger->warning('Gmail get label by name action: Label "@name" not found.', [
        '@name' => $label_name,
      ]);
      return;
    }

    $result = [
      'found' => true,
      'label_name' => $label_name,
      'id' => $label_details['id'],
      'name' => $label_details['name'],
      'type' => $label_details['type'],
      'messages_total' => $label_details['messages_total'] ?? 0,
      'messages_unread' => $label_details['messages_unread'] ?? 0,
      'threads_total' => $label_details['threads_total'] ?? 0,
      'threads_unread' => $label_details['threads_unread'] ?? 0,
    ];

    // Store result in token if requested
    if (!empty($label_token_name)) {
      $this->tokenService->addTokenData($label_token_name, $result);
    }
  }

}
