<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a specific message from Gmail.
 */
#[Action(
  id: 'eca_gmail_get_message',
  label: new TranslatableMarkup('Gmail: Get Message'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve specific email content, headers, and attachments with configurable format options for data extraction and processing workflows.'),
  version_introduced: '1.0.0',
)]
final class GetMessage extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'message_id' => '',
      'format' => 'full',
      'message_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['message_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message ID'),
      '#description' => $this->t('Gmail message ID to retrieve. Supports token replacement.'),
      '#default_value' => $this->configuration['message_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['format'] = [
      '#type' => 'select',
      '#title' => $this->t('Message Format'),
      '#description' => $this->t('Level of detail to retrieve for the message.'),
      '#default_value' => $this->configuration['format'],
      '#options' => $this->gmailService->getMessageFormatOptions(),
      '#required' => TRUE,
    ];

    $form['message_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message Token Name'),
      '#description' => $this->t('Optional token name to store message information. Accessible as [token_name:subject], [token_name:from], [token_name:body_html], [token_name:body_plain], etc.'),
      '#default_value' => $this->configuration['message_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $message_id = $this->tokenService->replacePlain($this->configuration['message_id']);
    $format = $this->configuration['format'];
    $message_token_name = $this->configuration['message_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get message');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($message_id)) {
      $this->logger->error('Gmail get message action: Missing message ID.');
      return;
    }

    // Get the message
    $message_data = $this->gmailService->getMessage($auth_type, $client_id, $message_id, $format);
    if (!$message_data) {
      $this->logger->error('Gmail get message action: Failed to get message @message_id using @auth_type.', [
        '@message_id' => $message_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store message data in token if requested
    if (!empty($message_token_name)) {
      $this->tokenService->addTokenData($message_token_name, $message_data);
    }
  }

}
