<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a Gmail thread with all messages.
 */
#[Action(
  id: 'eca_gmail_get_thread',
  label: new TranslatableMarkup('Gmail: Get Thread'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve entire email conversations with all messages for customer service workflows and conversation tracking.'),
  version_introduced: '1.0.0',
)]
final class GetThread extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'thread_id' => '',
      'format' => 'full',
      'thread_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['thread_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Thread ID'),
      '#description' => $this->t('Gmail thread ID to retrieve. Supports token replacement.'),
      '#default_value' => $this->configuration['thread_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['format'] = [
      '#type' => 'select',
      '#title' => $this->t('Message Format'),
      '#description' => $this->t('Level of detail to retrieve for each message in the thread. Note: RAW format is not available for threads.'),
      '#default_value' => $this->configuration['format'],
      '#options' => $this->gmailService->getThreadFormatOptions(),
      '#required' => TRUE,
    ];

    $form['thread_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Thread Token Name'),
      '#description' => $this->t('Optional token name to store thread information. Accessible as [token_name:message_count], [token_name:participants], [token_name:messages:0:subject], etc.'),
      '#default_value' => $this->configuration['thread_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $thread_id = $this->tokenService->replacePlain($this->configuration['thread_id']);
    $format = $this->configuration['format'];
    $thread_token_name = $this->configuration['thread_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('get thread');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($thread_id)) {
      $this->logger->error('Gmail get thread action: Missing thread ID.');
      return;
    }

    // Get the thread
    $thread_data = $this->gmailService->getThread($auth_type, $client_id, $thread_id, $format);
    if (!$thread_data) {
      $this->logger->error('Gmail get thread action: Failed to get thread @thread_id using @auth_type.', [
        '@thread_id' => $thread_id,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store thread data in token if requested
    if (!empty($thread_token_name)) {
      $this->tokenService->addTokenData($thread_token_name, $thread_data);
    }
  }

}
