<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\eca_google\Plugin\Action\GoogleApiActionBase;
use Drupal\eca_gmail\GmailService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Gmail actions.
 */
abstract class GmailActionBase extends GoogleApiActionBase {

  /**
   * The Gmail service.
   */
  protected GmailService $gmailService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->gmailService = $container->get('eca_gmail.gmail');
    return $instance;
  }

  /**
   * Parses email addresses from string.
   */
  protected function parseEmailAddresses(string $addresses): array {
    if (empty($addresses)) {
      return [];
    }

    // Split by comma and clean up
    $emails = array_map('trim', explode(',', $addresses));
    return array_filter($emails);
  }

  /**
   * Validates email address format.
   */
  protected function validateEmailAddress(string $email): bool {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== FALSE;
  }

  /**
   * Parses message IDs from string or array.
   */
  protected function parseMessageIds($message_ids): array {
    if (is_array($message_ids)) {
      return array_filter(array_map('trim', $message_ids));
    }

    if (is_string($message_ids)) {
      return array_filter(array_map('trim', explode(',', $message_ids)));
    }

    return [];
  }

  /**
   * Parses label IDs from string or array.
   */
  protected function parseLabelIds($label_ids): array {
    if (is_array($label_ids)) {
      return array_filter(array_map('trim', $label_ids));
    }

    if (is_string($label_ids)) {
      return array_filter(array_map('trim', explode(',', $label_ids)));
    }

    return [];
  }


}
