<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * List all Gmail labels with their IDs.
 */
#[Action(
  id: 'eca_gmail_list_labels',
  label: new TranslatableMarkup('Gmail: List Labels'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve all Gmail labels with their internal IDs for use in other Gmail actions like Apply Labels.'),
  version_introduced: '1.0.0',
)]
final class ListLabels extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'labels_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['labels_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Labels Token Name'),
      '#description' => $this->t('Optional token name to store label information. Accessible as [token_name:labels:0:id], [token_name:labels:0:name], [token_name:system_labels], [token_name:user_labels], etc.'),
      '#default_value' => $this->configuration['labels_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $labels_token_name = $this->configuration['labels_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('list labels');
    if (!$auth_type || !$client_id) {
      return;
    }

    // List labels
    $labels_result = $this->gmailService->listLabels($auth_type, $client_id);
    if (!$labels_result) {
      $this->logger->error('Gmail list labels action: Failed to retrieve labels using @auth_type.', [
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store labels in token if requested
    if (!empty($labels_token_name)) {
      $this->tokenService->addTokenData($labels_token_name, $labels_result);
    }
  }

}
