<?php

declare(strict_types=1);

namespace Drupal\eca_gmail\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Manage Gmail messages (archive, trash, restore, delete).
 */
#[Action(
  id: 'eca_gmail_manage_messages',
  label: new TranslatableMarkup('Gmail: Manage Messages'),
  category: new TranslatableMarkup('Gmail'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Archive, trash, restore, or permanently delete emails with support for bulk operations and automated housekeeping workflows.'),
  version_introduced: '1.0.0',
)]
final class ManageMessages extends GmailActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'message_ids' => '',
      'action' => 'archive',
      'management_token_name' => '',
    ] + $this->getAuthClientIdDefaultConfig()
      + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->addGoogleAuthConfigurationForm($form, $form_state);

    $form['message_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message IDs'),
      '#description' => $this->t('Gmail message IDs to manage (comma-separated). Supports token replacement.'),
      '#default_value' => $this->configuration['message_ids'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['action'] = [
      '#type' => 'select',
      '#title' => $this->t('Action'),
      '#description' => $this->t('Management action to perform on the messages.'),
      '#default_value' => $this->configuration['action'],
      '#options' => $this->gmailService->getMessageActionOptions(),
      '#required' => TRUE,
    ];

    $form['management_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Management Results Token Name'),
      '#description' => $this->t('Optional token name to store management results. Accessible as [token_name:affected_messages], [token_name:action], [token_name:total_requested], etc.'),
      '#default_value' => $this->configuration['management_token_name'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $message_ids = $this->tokenService->replacePlain($this->configuration['message_ids']);
    $action = $this->configuration['action'];
    $management_token_name = $this->configuration['management_token_name'];

    // Validate authentication
    [$auth_type, $client_id] = $this->validateAndParseAuth('manage messages');
    if (!$auth_type || !$client_id) {
      return;
    }

    if (empty($message_ids)) {
      $this->logger->error('Gmail manage messages action: Missing message IDs.');
      return;
    }

    // Parse message IDs
    $parsed_message_ids = $this->parseMessageIds($message_ids);
    if (empty($parsed_message_ids)) {
      $this->logger->error('Gmail manage messages action: No valid message IDs provided.');
      return;
    }

    // Prepare management parameters
    $manage_params = [
      'message_ids' => $parsed_message_ids,
      'action' => $action,
    ];

    // Manage messages
    $management_data = $this->gmailService->manageMessages($auth_type, $client_id, $manage_params);

    if (!$management_data) {
      $this->logger->error('Gmail manage messages action: Failed to @action messages using @auth_type.', [
        '@action' => $action,
        '@auth_type' => $auth_type,
      ]);
      return;
    }

    // Store management data in token if requested
    if (!empty($management_token_name)) {
      $this->tokenService->addTokenData($management_token_name, $management_data);
    }
  }

}
